/*-
 * Copyright (c) 2004 Networks Associates Technology, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * Forward declare libnfs opaque structures, as well as structures used by
 * node methods that are defined elsewhere.
 */
struct nfsdir;
struct nfsmount;
struct nfsnode;
struct stat;
struct statvfs;
struct timeval;

/*
 * struct nfscred describes a UNIX credential to be passed to the server
 * using the AUTH_SYS authentication type at the RPC layer.
 */
#define	NFSCRED_MAX_GROUPS	16
struct nfscred {
	uid_t	nc_uid;
	gid_t	nc_gid;
	u_int	nc_len;
	gid_t	nc_groups[NFSCRED_MAX_GROUPS];	/* XXXRW: Wrong constant? */
};

/*
 * struct nfsdirent describes an NFS directory entry, playing a similar role
 * to struct dirrent for the directory(3) API.  Uses the same DT_type
 * constants defined in dirent.h.  As with other NFS client/server
 * implementations, the d_type field is left 0'd.
 */
#define	NFSDIRENT_NAMELEN	256
struct nfsdirent {
	u_int32_t	d_fileno;
	u_int8_t	d_namlen;
	char		d_name[NFSDIRENT_NAMELEN];
};

/*
 * struct nfsinfo contains a local version of the information returned by
 * NFSPROC3_FSINFO, the static per-file system information provided by the
 * file server.  libnfs will read this information once on mount for each
 * nfsmount, then return the cached version from then on.
 */
#define	NFSINFO_CAP_LINK	0x0001
#define	NFSINFO_CAP_SYMLINK	0x0002
#define	NFSINFO_CAP_HOMOGENEOUS	0x0008
#define	NFSINFO_CAP_CANSETTIME	0x0010
struct nfsinfo {
	u_int32_t	ni_rtmax;		/* Maximum read size. */
	u_int32_t	ni_rtpref;		/* Preferred read size. */
	u_int32_t	ni_rtmult;		/* Suggested read multiple. */
	u_int32_t	ni_wtmax;		/* Maximum write size. */
	u_int32_t	ni_wtpref;		/* Preferred write size. */
	u_int32_t	ni_wtmult;		/* Suggesdted write multiple. */
	u_int32_t	ni_dtpref;		/* Preferred readdir size. */
	u_int64_t	ni_maxfilesize;		/* Maximum file size. */
	struct timespec	ni_timedelta;		/* Server time granularity. */
	u_int32_t	ni_properties;		/* NSFINFO_CAP flags. */
};

/*
 * NFS mount APIs, which both perform RPCs and setup/tear down data
 * structures.
 */
#define	NFSMOUNT_TYPE_PATH	1
#define	NFSMOUNT_TYPE_HANDLE	2
int	nfs_getroot(struct nfsmount *nfsm, struct nfsnode **nodep);
int	nfs_mount(struct nfsmount **nfsmp, struct nfscred *cred,
	    const char *hostname, int mounttype, const char *src,
	    const char *nettype);
int	nfs_setcred(struct nfsmount *nfsm, struct nfscred *cred);
void	nfs_unmount(struct nfsmount *nfsm);
int	nfs_null(struct nfsmount *nfsm);

char	*nfs_gethostname(struct nfsmount *nfsm);
char	*nfs_getmountpath(struct nfsmount *nfsm);

/*
 * Node reference management: APIs such as nfs_lookup() return references.
 */
void	nfs_node_deref(struct nfsnode *node);
void	nfs_node_ref(struct nfsnode *node);

/*
 * NFS node operations -- require a valid NFS node reference.
 */
int	nfs_getattr(struct nfsmount *nfsm, struct nfsnode *node,
	    struct stat *sbp);
int	nfs_getattr_cached(struct nfsmount *nfsm, struct nfsnode *node,
	    struct stat *sbp);
int	nfs_chmod(struct nfsmount *nfsm, struct nfsnode *node, mode_t mode);
int	nfs_chown(struct nfsmount *nfsm, struct nfsnode *node, uid_t uid,
	    gid_t gid);
int	nfs_truncate(struct nfsmount *nfsm, struct nfsnode *node,
	    off_t length);
int	nfs_utimes(struct nfsmount *nfsm, struct nfsnode *node,
	    struct timeval *times);
int	nfs_lookup(struct nfsmount *nfsm, struct nfsnode *dirnode,
	    char *name, struct nfsnode **nodep);
int	nfs_access(struct nfsmount *nfsm, struct nfsnode *node, int mode);
int	nfs_readlink(struct nfsmount *nfsm, struct nfsnode *node, char *buf,
	    int bufsiz);
ssize_t	nfs_read(struct nfsmount *nfsm, struct nfsnode *node, off_t offset,
	    void *buf, size_t nbytes);
ssize_t	nfs_write(struct nfsmount *nfsm, struct nfsnode *node, off_t offset,
	    void *buf, size_t nbytes);
int	nfs_create(struct nfsmount *nfsm, struct nfsnode *dirnode,
	    char *name, mode_t mode, struct nfsnode **nodep);
int	nfs_mkdir(struct nfsmount *nfsm, struct nfsnode *dirnode, char *name,
	    mode_t mode, struct nfsnode **nodep);
int	nfs_symlink(struct nfsmount *nfsm, struct nfsnode *dirnode,
	    char *name, char *target);
int	nfs_unlink(struct nfsmount *nfsm, struct nfsnode *dirnode,
	    char *name);
int	nfs_rmdir(struct nfsmount *nfsm, struct nfsnode *dirnode, char *name);
int	nfs_rename(struct nfsmount *nfsm, struct nfsnode *dirnode1,
	    char *name1, struct nfsnode *dirnode2, char *name2);
int	nfs_link(struct nfsmount *nfsm, struct nfsnode *node,
	    struct nfsnode *dirnode, char *name);
int	nfs_opendir(struct nfsmount *nfsm, struct nfsnode *dirnode,
	    struct nfsdir **dirp);
int	nfs_readdir_r(struct nfsdir *dir, struct nfsdirent *dirent,
	    struct nfsdirent **direntp);
void	nfs_closedir(struct nfsdir *dir);
int	nfs_fsinfo(struct nfsmount *nfsm, struct nfsinfo *info);
int	nfs_mknod(struct nfsmount *nfsm, struct nfsnode *dirnode, char *name,
	    mode_t nodetype, mode_t mode, u_int32_t major, u_int32_t minor,
	    struct nfsnode **nodep);
int	nfs_fsync(struct nfsmount *nfsm, struct nfsnode *node);
int	nfs_statvfs(struct nfsmount *nfsm, struct statvfs *buf);

/*
 * These NFS operations are not yet prototyped fully because they are not yet
 * implemented.
 */
int	nfs_pathconf(struct nfsmount *nfsm);
