/*-
 * Copyright (c) 2004 Networks Associates Technology, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/mac.h>
#include <sys/mount.h>
#include <sys/stat.h>

#include <errno.h>
#include <fcntl.h>
#include <syslog.h>

#include "nfs_label.h"

static enum nfs3_label_status
error_to_nfs3labelstatus(int error)
{

	switch (error) {
	case EPERM:
		return (NFS3_LABEL_PERM);
	case EIO:
		return (NFS3_LABEL_IO);
	case ENOMEM:
		return (NFS3_LABEL_NOMEM);
	case EACCES:
		return (NFS3_LABEL_ACCES);
	case EOPNOTSUPP:
		return (NFS3_LABEL_OPNOTSUPP);
	case ESTALE:
		return (NFS3_LABEL_STALE);
	case ENOSYS:
		return (NFS3_LABEL_NOSYS);
	default:
		return (NFS3_LABEL_IO);		/* XXX: Right default? */
	}
}

static int
nfs3_label_fhopen(struct nfs3_label_fh3 *fh3, int flags)
{
	fhandle_t fhandle;
	int fd;

	/*
	 * We map the NFSv3 distributed file handle into a local file handle
	 * structure.  We reject handles that don't fit the native format for
	 * handles.
	 */
	if (fh3->data.data_len != sizeof(fhandle)) {
		errno = ESTALE;			/* XXX: Right error? */
		return (-1);
	}

	bcopy(fh3->data.data_val, &fhandle, sizeof(fhandle));
	return (fhopen(&fhandle, flags));
}

void *
nfsproc3_label_null_3_svc(void *argp, struct svc_req *rqstp)
{
	static char * result;

	bzero(&result, sizeof(result));
	result = NULL;
	return((void *) &result);
}

GETLABEL3res *
nfsproc3_label_getlabel_3_svc(GETLABEL3args *argp, struct svc_req *rqstp)
{
	static GETLABEL3res  result;
	static char *text;
	mac_t label;
	int fd;

	/*
	 * Because 'text' must be static to last after return, we may have to
	 * free the previous instance before starting.
	 */
	if (text != NULL) {
		free(text);
		text = NULL;
	}

	bzero(&result, sizeof(result));

	fd = nfs3_label_fhopen(&argp->object, O_RDONLY);
	if (fd == -1) {
		result.status = error_to_nfs3labelstatus(errno);
		return (&result);
	}

	if (mac_prepare(&label, argp->elements) == -1) {
		result.status = error_to_nfs3labelstatus(errno);
		return (&result);
	}

	if (mac_get_fd(fd, label) == -1) {
		result.status = error_to_nfs3labelstatus(errno);
		mac_free(label);
		return (&result);
	}

	if (mac_to_text(label, &text) == -1) {
		result.status = error_to_nfs3labelstatus(errno);
		mac_free(label);
		text = NULL;
		return (&result);
	}

	if (strlen(text) > NFS3_LABEL_MAXSTRINGLEN) {
		result.status = error_to_nfs3labelstatus(ENOMEM);
		free(text);
		text = NULL;
		mac_free(label);
		return (&result);
	}
	result.GETLABEL3res_u.resok.label = text;

	/* Note that text is not freed, see comment above. */
	mac_free(label);
	close(fd);
	result.status = NFS3_LABEL_OK;
	return (&result);
}

SETLABEL3res *
nfsproc3_label_setlabel_3_svc(SETLABEL3args *argp, struct svc_req *rqstp)
{
	static SETLABEL3res  result;
	int fd;

	bzero(&result, sizeof(result));

	fd = nfs3_label_fhopen(&argp->object, O_RDONLY);
	if (fd == -1) {
		result.status = error_to_nfs3labelstatus(errno);
		return (&result);
	}
	close(fd);

	result.status = NFS3_LABEL_PERM;
	return (&result);
}
