/*-
 * Copyright (c) 2003 Networks Associates Technology, Inc.
 * All rights reserved.
 *
 * This software was developed for the FreeBSD Project by and Network
 * Associates Laboratories, the Security Research Division of Network
 * Associates, Inc. under DARPA/SPAWAR contract N66001-01-C-8035 ("CBOSS"),
 * as part of the DARPA CHATS research program.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */
#include <sys/types.h>
#include <sys/disk.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "diskscan.h"

int jflag;
int vflag;

static void
usage(void)
{

	fprintf(stderr, "diskscan [-fjv] [-c sectorcount] [-s sectorsize] "
	    "device testname [testargs ...]\n");
	fprintf(stderr, "Available tests: read, readwrite, write\n");
	exit(-1);
}

/*
 * Eventually, we might support pluggable modules here, but for now,
 * just convert a string holding the test name to a statically linked
 * symbol for the test description.
 */
const struct disk_test *
disk_test_lookup(const char *name)
{

	if (strcmp(name, "read") == 0)
		return (&disk_test_read);
	if (strcmp(name, "write") == 0)
		return (&disk_test_write);
	if (strcmp(name, "readwrite") == 0)
		return (&disk_test_readwrite);
	if (strcmp(name, "wrwr") == 0)
		return (&disk_test_wrwr);
	return (NULL);
}

int
main(int argc, char *argv[])
{
	int ch, devicefd, error, fflag, sectorsize;
	char *devicename, *forced_sectorcount, *forced_sectorsize, *testname;
	const struct disk_test *dt;
	off_t sectorcount;
	struct stat sb;

	forced_sectorcount = NULL;
	forced_sectorsize = NULL;
	fflag = 0;
	while ((ch = getopt(argc, argv, "c:fjs:v")) != -1)
		switch (ch) {
		case 'c':
			forced_sectorcount = optarg;
			break;
		case 'f':
			fflag = 1;
			break;
		case 'j':
			jflag = 1;
			break;
		case 's':
			forced_sectorsize = optarg;
			break;
		case 'v':
			vflag = 1;
			break;
		default:
			usage();
		}
	argc -= optind;
	argv += optind;

	if (argc < 2)
		usage();

	devicename = argv[0];
	testname = argv[1];

	argc -= 2;
	argv += 2;

	dt = disk_test_lookup(testname);
	if (dt == NULL) {
		fprintf(stderr, "diskscan: test '%s' unknown\n", testname);
		exit(-1);
	}
	if (vflag)
		printf("Test: %s\n", testname);

	devicefd = open(devicename, dt->dt_openmode, 0000);
	if (devicefd == -1) {
		perror(devicename);
		exit(-1);
	}
	if (vflag)
		printf("Device: %s\n", devicename);

	if (forced_sectorsize != NULL) {
		/*
		 * Additional error checking here would be useful.
		 */
		sectorsize = atoi(forced_sectorsize);
		if (sectorsize == 0) {
			fprintf(stderr, "Invalid sector size: %d\n",
			    sectorsize);
			exit(-1);
		}
	} else if (fflag)
		sectorsize = 1;
	else {
		error = ioctl(devicefd, DIOCGSECTORSIZE, &sectorsize);
		if (error) {
			perror("DIOCGSECTORSIZE");
			exit(-1);
		}
	}
	if (vflag)
		printf("Sectorsize: %d\n", sectorsize);

	if (forced_sectorcount)
		sectorcount = atoi(forced_sectorcount);
	else if (fflag) {
		error = fstat(devicefd, &sb);
		if (error) {
			perror("stat");
			exit(-1);
		}
		/*
		 * NOTE: we truncate to the nearest sectorsize bytes on
		 * files that are no even multiples of sectorsize bytes.
		 */
		sectorcount = sb.st_size / sectorsize;
	} else {
		error = ioctl(devicefd, DIOCGMEDIASIZE, &sectorcount);
		if (error) {
			perror("DIOCGMEDIASIZE");
			exit(-1);
		}
		sectorcount /= sectorsize;
	}
	if (vflag)
		printf("Sectorcount: %jd\n", sectorcount);

	return ((dt->dt_function)(devicefd, devicename, sectorsize,
	    sectorcount, argc, argv));
}
