/*-
 * Copyright (c) 2004 Networks Associates Technology, Inc.
 * All rights reserved.
 *
 * This software was developed by McAfee Research under DARPA/AFRL contract
 * number F30602-00-C-0183 in support of VCC seedling research.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */

#include <sys/types.h>

#include <dlfcn.h>
#include <md5.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ucontext.h>
#include <unistd.h>

#include "ddb.h"

struct i386_frame {
	struct i386_frame	*f_frame;
	void			*f_retaddr;
	u_int32_t		 f_arg0;
};

struct frame {
	struct i386_frame	*f_prev;
	struct i386_frame	*f_this;
	ucontext_t		*f_context;
	void			*f_ip;
	int64_t			 f_size;
	int			 f_valid;
};

/*
 * Global context for analysis following a live snapshot to avoid
 * putting it on the stack.
 */
#define	STACK_SIZE	65536
static ucontext_t snapshot_context;

static void
context_init(ucontext_t *context)
{

	context->uc_stack.ss_sp = malloc(STACK_SIZE);
	if (context->uc_stack.ss_sp == NULL) {
		perror("ddb_init");
		exit(-1);
	}
	context->uc_stack.ss_size = STACK_SIZE;
	context->uc_link = NULL;
}

void
ddb_init(void)
{

	getcontext(&snapshot_context);
	context_init(&snapshot_context);
}

static void
dumpbytes(void *ptr, u_int32_t len)
{
	u_int32_t i, j;

	for (j = 0; j < len / 4 + 1; j++) {
		if (j % 8 == 0)
			printf("\n    ");
		for (i = 0; i < 4; i++) {
			if (j * 4 + i < len)
				printf("%02x", *((u_char *)ptr + (j * 4 + i)));
		}
		if (j % 8 != 7)
			printf(" ");
	}
	printf("\n");
}

struct frame
next_frame(ucontext_t *context, struct frame prev_frame)
{
	struct frame frame;

	bzero(&frame, sizeof(frame));
	frame.f_prev = prev_frame.f_this;
	frame.f_this = frame.f_prev->f_frame;
	if (frame.f_this == NULL) {
		frame.f_valid = 0;
		return (frame);
	}
	frame.f_context = prev_frame.f_context;
	frame.f_ip = prev_frame.f_this->f_retaddr;
	frame.f_valid = 1;
	frame.f_size = (int64_t)(u_int32_t)frame.f_this -
	    (int64_t)(u_int32_t)frame.f_prev;
	if (frame.f_size < 0)
		frame.f_size = -1;

	return (frame);
}

struct frame
first_frame(ucontext_t *context)
{
	struct i386_frame *i386_frame;
	struct frame frame;

	printf("first_frame: esp 0x%08x ebp 0x%08x\n",
	    (u_int32_t)context->uc_mcontext.mc_esp,
	    (u_int32_t)context->uc_mcontext.mc_esp);
	i386_frame = ((void *)context->uc_mcontext.mc_esp - 4);
	bzero(&frame, sizeof(frame));
	frame.f_prev = i386_frame;
	frame.f_this = frame.f_prev->f_frame;
	frame.f_context = context;
	frame.f_ip = i386_frame->f_retaddr;
	frame.f_valid = 1;
	frame.f_size = (int64_t)(u_int32_t)frame.f_this -
	    (int64_t)(u_int32_t)frame.f_prev;
	if (frame.f_size < 0)
		frame.f_size = -1;

	/* Return the frame after the debugger frame. */
	return (next_frame(context, frame));
}

static void
trace_context(ucontext_t *context, int skipframes, int tracestyle, int flags)
{
	int symbol_needed, symbol_found, active_tracestyle;
	struct frame frame;
	int count, skipped;
	Dl_info symbol;

	printf("Stacktrace of context 0x%08x stack 0x%08x:\n",
	    (u_int32_t)context, (u_int32_t)(context->uc_mcontext.mc_esp));
	frame = first_frame(context);
	if (!frame.f_valid) {
		printf("  Invalid first frame\n");
		return;
	}

	skipped = 0;
	count = 1;
	while (frame.f_valid) {
		if (skipped++ >= skipframes) {
			switch (tracestyle) {
			case DDB_BORING:
				symbol_needed = 0;
				break;
			case DDB_FUNCTIONCENTRIC:
			case DDB_OBJECTCENTRIC:
			default:
				symbol_needed = 1;
			}

			if (symbol_needed)
				symbol_found = dladdr(frame.f_ip, &symbol);
			else
				symbol_found = 0;

			if (symbol_needed && !symbol_found)
				active_tracestyle = DDB_BORING;
			else
				active_tracestyle = tracestyle;
			switch (active_tracestyle) {
			case DDB_BORING:
				printf("  #%d 0x%x\n", count,
				    (u_int32_t)frame.f_ip);
				break;
			case DDB_OBJECTCENTRIC:
				printf("  #%d %s+0x%x (0x%08x)\n", count,
				    symbol.dli_fname, (u_int32_t)frame.f_ip -
				    (u_int32_t)symbol.dli_fbase,
				    (u_int32_t)frame.f_ip);
				break;

			case DDB_FUNCTIONCENTRIC:
			default:
				printf("  #%d %s %s+0x%x (0x%08x)\n", count,
				    symbol.dli_fname, symbol.dli_sname,
				    (u_int32_t)frame.f_ip -
				    (u_int32_t)symbol.dli_saddr,
				    (u_int32_t)frame.f_ip);

			}

			if (flags & DDB_DUMPFRAME && frame.f_size >= 0) {
				dumpbytes((char *)frame.f_this -
				    sizeof(*frame.f_this), frame.f_size);
				printf("\n");
			}
		}
		if (frame.f_size < 0) {
			if (flags & DDB_FAILSTOP) {
				fprintf(stderr, "FAILSTOP\n");
				exit(-1);
			} else
				break;
		}
		count++;
		frame = next_frame(context, frame);
	}
}

void
simpletrace(void)
{
	ucontext_t context;
	struct frame frame;
	int count;

	getcontext(&context);
	frame = first_frame(&context);
	count = 1;
	while (frame.f_valid) {
		printf("#%d 0x%08x\n", count, (u_int32_t)frame.f_ip);
		count++;
		frame = next_frame(&context, frame);
	}
}

void
ddb_stacktrace(int style, int flags)
{

	getcontext(&snapshot_context);
	trace_context(&snapshot_context, 0, style, flags);
}

void
ddb_stacktrace_context(ucontext_t *context, int style, int flags)
{

	trace_context(context, 0, style, flags);
}
