/*-
 * Copyright (c) 1998 Robert N. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name Robert N. Watson may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: ktoken.c,v 1.9 1999/01/12 15:28:02 robert Exp $
 */

/*
 * ktoken syscall implementation, kernel event hooks (fork, exit)
 */


#include <sys/param.h>
#include <sys/systm.h>
#include <sys/exec.h>
#include <sys/sysent.h>
#include <sys/lkm.h>
#include <sys/malloc.h>
#include <sys/errno.h>
#include <sys/proc.h>
#include <sys/mbuf.h>
#include <sys/socket.h>

#include "../common/ktoken_const.h"
#include "../common/ktoken_types.h"
#include "../common/ktoken_structs.h"
#include "../common/ktoken_syscall.h"
#include "ktoken_kconst.h"
#include "ktoken_kstructs.h"
#include "ktoken.h"
#include "ktoken_support.h"

extern int ktoken_syscall __P((struct proc *p, struct t_syscall_args *uap));
#ifdef KTOKEN_TRANSFER
extern int ktoken_unp_externalize __P((struct mbuf *control, struct proc *p));
extern int ktoken_unp_internalize __P((struct mbuf *control, struct proc *p));
#endif

/* -stable doesn't have setsuigid() */
#ifndef setsugid
#define setsugid(p) { p->p_flag |= P_SUGID; }
#endif

/* globals */
int	ktdebug;
static pagid_t		badpagid=0;


/* syscall multiplex -- this is the real syscall */
int
ktoken_syscall(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    int fn = uap->whichfunction;
    KTDEBUG(2,"ktoken_syscall(): function %d\n", uap->whichfunction);
    
    /* check validity of argument */
    if (
	!((fn >= TOKEN_CONTROL_MIN) &&
	  (fn <= TOKEN_CONTROL_MAX))
	&&
	!((fn >= TOKEN_PAGCALL_MIN) &&
	  (fn <= TOKEN_PAGCALL_MAX))
	&&
	!((fn >= TOKEN_TOKCALL_MIN) &&
	  (fn <= TOKEN_TOKCALL_MAX))
	&&
	!((fn >= TOKEN_USE_MIN) &&
	  (fn <= TOKEN_USE_MAX))
	)
    {
	KTDEBUG(1,"ktoken_syscall(): invalid function call\n");
	return(EINVAL);
    }
    switch(fn) {
    case TOKEN_SYSCALL_SETDEBUG:
	KTDEBUG(2,"ktoken_syscall.ktoken_setdebug()\n");
	return ktoken_setdebug(p, uap);
    case TOKEN_SYSCALL_SETTOKEND:
	KTDEBUG(2, "ktoken_syscall.ktoken_settokend()\n");
	return ktoken_settokend(p, uap);
    case TOKEN_SYSCALL_NEWPAG:
	KTDEBUG(2,"ktoken_syscall.ktoken_newpag()\n");
	return ktoken_newpag(p, uap);
    case TOKEN_SYSCALL_GETPAG:
	KTDEBUG(2,"ktoken_syscall.ktoken_getpag()\n");
	return ktoken_getpag(p, uap);
    case TOKEN_SYSCALL_READTOKEN:
	KTDEBUG(2,"ktoken_syscall.ktoken_readtoken()\n");
	return ktoken_readtoken(p, uap);
    case TOKEN_SYSCALL_MODIFYTOKEN:
	KTDEBUG(2,"ktoken_syscall.ktoken_modifytoken()\n");
	return ktoken_modifytoken(p, uap);
    case TOKEN_SYSCALL_CREATETOKEN:
	KTDEBUG(2,"ktoken_syscall.ktoken_createtoken()\n");
	return ktoken_createtoken(p, uap);
    case TOKEN_SYSCALL_REFLECTTOKEN:
	KTDEBUG(2,"ktoken_syscall.ktoken_reflecttoken()\n");
	return ktoken_reflecttoken(p, uap);
    case TOKEN_SYSCALL_DELETETOKEN:
	KTDEBUG(2,"ktoken_syscall.deletetoken()\n");
	return ktoken_deletetoken(p, uap);
    case TOKEN_SYSCALL_FINDTOKEN:
	KTDEBUG(2,"ktoken_syscall.findtoken()\n");
	return ktoken_findtoken(p, uap);
    case TOKEN_SYSCALL_SETUIDTOKEN:
	KTDEBUG(2,"ktoken_syscall.setuidtoken()\n");
	return ktoken_setuidtoken(p, uap);
    default:
	KTDEBUG(0,"ktoken_syscall(): invalid function beyond check\n");
	return(EINVAL);
    }
    
    /* never get here */
    return(EINVAL);		/* success (or error code from errno.h)*/
}

/* macros to copy typed data from/into user space */
#define T_COPYIN(src, dest, type) copyin(src, dest, sizeof(type))
#define T_COPYOUT(src, dest, type) copyout(src, dest, sizeof(type))

/* make this process part of a new pag */
static int
ktoken_newpag(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_newpag_args a;
    int i;
    
    struct pagstr *new, *old;
    
    i = T_COPYIN(uap->args, &a, struct t_newpag_args);
    if (i) return(i);
    
    /* find the current pag */
    t_internal_find_pag_by_proc(&old, p);
    
    /* create a new pag */
    t_internal_newpag(&new, M_WAITOK);
    if (!new) {
	/* t_internal_return_pagstr(old); */
	KTDEBUG(1, "ktoken.ktoken_newpag(): t_internal_newpag() failed\n");
	return(EAGAIN);
    }
    
    /* if there was an old one, inherit its tokens, but leave it */
    if (old) {
	t_internal_inheritpag(old, new);
	t_internal_remove_proc_pag(p, old);
    }
    /* add the current proc to the new pag */
    t_internal_add_proc_pag(p, new);
    
    /* return new pagid */
    i = T_COPYOUT(&(new->pagid), a.p_pagid, pagid_t);
    return(i);
}

/* return the current process' pagid */
static int
ktoken_getpag(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_getpag_args a;
    struct pagstr *p_pag;
    int i;
    
    i = T_COPYIN(uap->args, &a, struct t_getpag_args);
    if (i) return(i);
    
    /* find the current pag */
    t_internal_find_pag_by_proc(&p_pag, p);
    
    if (!p_pag) {
	return(T_PAGNOTFOUND);
    }
    
    /* return the pagid */
    i = T_COPYOUT(&(p_pag->pagid), a.p_pagid, pagid_t);
    return(i);
}

/* return a copy of an identified token */
static int
ktoken_readtoken(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_readtoken_args a;
    struct pagstr *p_pag;
    struct ktokenstr *p_token;
    struct utokenstr outtoken;	/* is this too big for stack? */
    int i;
    
    i = T_COPYIN(uap->args, &a, struct t_readtoken_args);
    if (i) return i;
    
    /* find the current pag */
    t_internal_find_pag_by_proc(&p_pag, p);
    if (!p_pag) {
	return (T_PAGNOTFOUND);
    }
    
    /* find the appropriate token */
    t_internal_find_token_by_tokenid(a.tokenid, &p_token);
    if (!p_token) {
	return(T_TOKENNOTFOUND);
    }
    
    /* create an appropriate-looking copy of the tokens */
    i = t_internal_read_token(p_pag, p_token, &outtoken);
    
    if (i) {
	return(i);
    }
    
    /* return contents of token */
    i = T_COPYOUT(&outtoken, a.p_token, struct utokenstr);
    return(i);
}

/* change specific fields of a token */
static int
ktoken_modifytoken(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_modifytoken_args a;
    struct pagstr *p_pag;
    struct utokenstr modeltoken;
    struct ktokenstr *p_token;
    int i;
    
    i = T_COPYIN(uap->args, &a, struct t_modifytoken_args);
    if (i) return(i);
    
    /* the token field model */
    i = T_COPYIN(a.p_token, &modeltoken, struct utokenstr);
    if (i) return(i);
    
    /* find the current pag */
    t_internal_find_pag_by_proc(&p_pag, p);
    if (!p_pag) {
	return (T_PAGNOTFOUND);
    }
    
    /* find the appropriate destination token */
    t_internal_find_token_by_tokenid(a.tokenid, &p_token);
    if (!p_token) {
	/* t_internal_return_pagstr(p_pag); */
	return (T_TOKENNOTFOUND);
    }
    
    /* modify the token in the "process" role */
    i = t_internal_modify_token(TOKEN_CRT_PROCESS, p_pag, p_token,
				&modeltoken, a.whichfields);
    
    return(i);
}

/* create a new token from model */
static int 
ktoken_createtoken(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_createtoken_args a;
    struct pagstr *p_pag;
    struct utokenstr modeltoken; /* XXXX stack space? */
    struct ktokenstr *outtoken;
    int i, role;
    
    i = T_COPYIN(uap->args, &a, struct t_createtoken_args);
    if (i) return(i);
    
    /* token to create from */
    i = T_COPYIN(a.p_token, &modeltoken, struct utokenstr);
    if (i) return(i);
    
    /* find current PAG */
    t_internal_find_pag_by_proc(&p_pag, p);
    if (!p_pag) {
	return (T_PAGNOTFOUND);
    }

    /* determine the role to create the token as */
    if (p_pag->tokend_count) {
	KTDEBUG(5, "ktoken.ktoken_createtoken(): has non-zero tokend_count\n");
	/* may have a TOKEND localpriv token */
	if (t_internal_pag_has_tokend(p_pag)) {
	    KTDEBUG(5, "ktoken.ktoken_createtoken(): using role TOKEND\n");
	    role = TOKEN_CRT_TOKEND;
	} else {
	    KTDEBUG(5, "ktoken.ktoken_createtoken(): using role PROCESS\n");
	    role = TOKEN_CRT_PROCESS;
	}
    } else {
	role = TOKEN_CRT_PROCESS;   /* normal process */
	KTDEBUG(5,  "ktoken.ktoken_createtoken(): using role PROCESS\n");
    }
    
    /* create the token in "process" role */
    i = t_internal_create_token(role, p_pag, &modeltoken,
				&outtoken);
    if (i) return(i);
    
    /* add the token to the current PAG */
    i = t_internal_add_token_to_pag(outtoken, p_pag);
    if (i) {
	/* but if we can't, destroy the token */
	outtoken->refcount++; /* XXXX HACK to make gc'able */
	t_internal_decref_tokenstr(outtoken);
	return(i);
    }
    
    /* return tokenid */
    i = T_COPYOUT(&(outtoken->utoken.tokenid), a.p_tokenid, tokenid_t);
    return(i);
}

/* create a reflection of a token */
static int 
ktoken_reflecttoken(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_reflecttoken_args a;
    struct pagstr *p_pag;
    struct ktokenstr *p_token, *outtoken;
    int i;
    
    i = T_COPYIN(uap->args, &a, struct t_reflecttoken_args);
    if (i) return(i);
    
    /* find the current PAG */
    t_internal_find_pag_by_proc(&p_pag, p);
    if (!p_pag) {
	return (T_PAGNOTFOUND);
    }

    /* find the source token */
    t_internal_find_token_by_tokenid(a.tokenid1, &p_token);
    if (!p_token) {
	return (T_TOKENNOTFOUND);
    }
    
    /* generate reflection */
    i = t_internal_reflect_token(p_pag, p_token, &outtoken);
    if (i) {
	return (i);
    }
    
    /* add token to pag */
    i = t_internal_add_token_to_pag(outtoken, p_pag);
    if (i) {
	/* if we can't, destroy the reflection */
	p_token->refcount++; /* XXXX HACK to make gc'able */
	t_internal_decref_tokenstr(p_token);
	return(i);
    }
    
    /* return tokenid */
    i = T_COPYOUT(&(outtoken->utoken.tokenid), a.p_tokenid2, tokenid_t);
    return(i);
}

/* delete a token */
static int 
ktoken_deletetoken(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_deletetoken_args a;      
    struct pagstr *p_pag;
    struct ktokenstr *p_token;
    int i;
    
    i = T_COPYIN(uap->args, &a, struct t_deletetoken_args);
    if (i) return(i);
    
    /* find the current PAG */
    t_internal_find_pag_by_proc(&p_pag, p);
    if (!p_pag) {
	return (T_PAGNOTFOUND);
    }
    
    /* find the token to delete */
    t_internal_find_token_by_tokenid(a.tokenid, &p_token);
    if (!p_token) {
	return(T_TOKENNOTFOUND);
    }
    
    /* delete the token */
    i = t_internal_delete_token(TOKEN_CRT_PROCESS, p_pag, p_token);
    return(i);
}

/* find a token matching the specified fields of a sample token */
static int
ktoken_findtoken(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_findtoken_args a;
    struct pagstr *p_pag;
    struct ktokenstr *p_token;
    struct utokenstr modeltoken;  /* XXXX stack space? */
    tokenid_t tokenid;
    int i;
    
    i = T_COPYIN(uap->args, &a, struct t_findtoken_args);
    if (i) return(i);
    
    i = T_COPYIN(a.p_token, &modeltoken, struct utokenstr);
    if (i) return(i);

    /* find the current PAG */
    t_internal_find_pag_by_proc(&p_pag, p);
    if (!p_pag) {
	return (T_PAGNOTFOUND);
    }

    KTDEBUG(5, "ktoken.ktoken_findtoken(): pag %lu searching for at least "
	    "%lu, fields %du\n", p_pag->pagid, a.mintokenid, a.whichfields);

    /* perform the search */
    t_internal_search_token(p_pag,
			    a.mintokenid,
			    a.whichfields,
			    &modeltoken,
			    &p_token);
    if (!p_token) {
	/* not found */
	KTDEBUG(5, "ktoken.ktoken_findtoken(): pag %lu search turned "
		"up nothing\n", p_pag->pagid);
	tokenid = 0;
    } else {
	/* found */
	KTDEBUG(5, "ktoken.ktoken_findtoken(): pag %lu search returned token "
		"%lu\n", p_pag->pagid, p_token->utoken.tokenid);

	tokenid = p_token->utoken.tokenid;
    }

    /* return the tokenid */
    i = T_COPYOUT(&(tokenid), a.p_tokenid, tokenid_t);
    return(i);
}

/* enable/disable debugging level */
static int
ktoken_setdebug(p, uap)
struct proc *p;         
struct t_syscall_args *uap;
{       
        struct t_setdebug_args a;
        int i;

	/* only uid 0 may do this */
	if (p->p_cred->pc_ucred->cr_uid != 0)
		return(EPERM);
        
        i = T_COPYIN(uap->args, &a, struct t_setdebug_args);
        if (i) return(i);

	/* set level */
	ktdebug = a.debug;

	printf("ktoken.ktoken_debug(): debugging set to %d\n", ktdebug);

	return(0);
}

static int
ktoken_settokend(p, uap)
struct proc *p;
struct t_syscall_args *uap;
{
    struct t_settokend_args a;
    struct pagstr *p_pag;
    struct utokenstr u;
    struct ktokenstr *outtoken;
    int i;

    /* this is for development purposes only -- for now, require uid 0. */
    /* XXXX */
    if (p->p_cred->pc_ucred->cr_uid != 0)
	return(EPERM);
    
    bzero(&u, sizeof(struct utokenstr));
    u.major = TOKEN_MAJ_LOCALPRIV;
    u.minor = TOKEN_LPRIV_TOKEND;
    u.minorminor = 0;
    u.rights = T_R_DELETE;
    strncpy(u.name, "tokend", T_STRINGLEN_MAX);
    strncpy(u.realm, "localpriv", T_STRINGLEN_MAX);
    u.publiclen = u.privatelen = 0;

    t_internal_find_pag_by_proc(&p_pag, p);
    if (!p_pag) { 
	return (T_PAGNOTFOUND);
    }

    /* create the token in "kernel" role */
    i = t_internal_create_token(TOKEN_CRT_KERNEL, p_pag, &u, &outtoken);
    if (i) return(i);

    /* add the token to the current PAG */
    i = t_internal_add_token_to_pag(outtoken, p_pag);
    if (i) {
	/* but if we can't, destroy the token */
	outtoken->refcount++; /* XXXX HACK to make gc'able */
	t_internal_decref_tokenstr(outtoken);
	return(i);
    }
    return(0);
}

static int
ktoken_setuidtoken(struct proc *p,
		   struct t_syscall_args *uap) {
    struct t_setuidtoken_args a;
    struct pagstr *p_pag;
    struct ktokenstr *p_token;
    struct ktokenrefstr *p_tokenref;
    int i;
    uid_t uid;
    struct pcred *pc = p->p_cred;

    /* given an appropriate token, change the user's effective uid */
    
    i = T_COPYIN(uap->args, &a, struct t_setuidtoken_args);
    if (i) return(i);

    /* find the current PAG */
    t_internal_find_pag_by_proc(&p_pag, p);
    if (!p_pag) {
	KTDEBUG(2, "ktoken.ktoken_setuidtoken(): proc %d has no pag\n",
		p->p_pid);
	return (T_PAGNOTFOUND);
    }

    /* find the token in question */
    t_internal_find_token_by_tokenid(a.tokenid, &p_token);
    if (!p_token) {
	KTDEBUG(2, "ktoken.ktoken_setuidtoken(): token %lu does not exist\n",
		a.tokenid);
	return(T_TOKENNOTFOUND);
    }

    /* check that this PAG has access to that token */

    t_internal_pag_get_tokenref(p_pag, p_token, &p_tokenref);
    if (!p_tokenref) {
	KTDEBUG(2, "ktoken.ktoken_setuidtoken(): pag %lu does not have access "
		"to token %lu\n", p_pag->pagid, p_token->utoken.tokenid);
	return(EPERM);
    }
    
    /* has this token; now see if it is a good token for the request */
    if (t_internal_token_has_expired(p_token)) {
	KTDEBUG(2, "ktoken.ktoken_setuidtoken(): attempt to use expired token "
		"%lu\n", p_token->utoken.tokenid);
	return(T_TOKENEXPIRED);
    }

    if (!t_internal_token_is_real_localpriv(p_token)) {
	KTDEBUG(2, "ktoken.ktoken_setuidtoken(): attempt to use non-localpriv "
		"token %lu to setuid\n", p_token->utoken.tokenid);
	return(EPERM);
    }

    /* is a real, unexpired token that the user really has */
    /* check if it is a uid token */
    if (p_token->utoken.minor != TOKEN_LPRIV_UID) {
	return(EINVAL);
    }

    uid = p_token->utoken.minorminor;
    KTDEBUG(2, "ktoken.ktoken_setuidtoken(): set uid of proc %d to %d\n",
	    p->p_pid, uid);


    /* won't save effective uid, as that is for setuid, not setuidtoken */

    if (uid != pc->p_ruid) {
	(void) chgproccnt(pc->p_ruid, -1);
	(void) chgproccnt(uid, 1);
    }

    /* set real uid */
    if (uid != pc->p_ruid) {
	pc->p_ruid = uid;
	setsugid(p);
    }

    /* set saved uid -XXX should this happen? */
    if (pc->p_svuid != uid) {
	pc->p_svuid = uid;
	setsugid(p);
    }
    
    /* change the effective uid */
    if (pc->pc_ucred->cr_uid != uid) {
	pc->pc_ucred = crcopy(pc->pc_ucred);
	pc->pc_ucred->cr_uid = uid;
	setsugid(p);
    }

    return(0);
}


/* hook fork actions */
void
ktoken_at_fork(struct proc *parent,struct proc *child,int flags)
{
	struct pagstr *p_pag;
	KTDEBUG(5,"ktoken.ktoken_at_fork(): fork occurred from %d to %d "
		"w/flags %d\n", parent->p_pid, child->p_pid, flags);

	/* find the parent's PAG */
	t_internal_find_pag_by_proc(&p_pag, parent);

	if (p_pag) {
	    /* if found */
	    KTDEBUG(1, "ktoken.ktoken_at_fork(): %d has a pag (%lu), "
		    "so am adding child %d to it\n", parent->p_pid, 
		    p_pag->pagid, child->p_pid);

	    /* add child to that pag */
	    t_internal_add_proc_pag(child, p_pag);
	} else {
	    /* if not, nothing */
		KTDEBUG(5, "ktoken.ktoken_at_fork(): %d -> %d fork did not "
			"result in pag inheritence\n", parent->p_pid, 
			child->p_pid);
	}
}

/* hook process exit actions */
void
ktoken_at_exit(struct proc *procp)
{
    struct pagstr *p_pag;
    KTDEBUG(5,"ktoken.ktoken_at_exit(): exit occurred in %d\n",
	    procp->p_pid);

    /* find dying process' PAG */
    t_internal_find_pag_by_proc(&p_pag, procp);

    if (p_pag) {
	/* if found */
	KTDEBUG(1, "ktoken.ktoken_at_exit(): %d has a pag (%lu), so am "
		"removing it\n", procp->p_pid, p_pag->pagid);

	/* remove proc from PAG */
	t_internal_remove_proc_pag(procp, p_pag);
    } else {
	/* if not, nothing */
	KTDEBUG(5, "ktoken.ktoken_at_exit(): %d has no pag, so won't remove "
		"it\n", procp->p_pid);
    }
}

#ifdef KTOKEN_TRANSFER
/* These differ from prototypes to prevent requiring struct mbuf def in
   sys/systm.h.  */

/* Called to unpack a transfered token at the recipient (p) */
int ktoken_unp_externalize (struct mbuf *control, struct proc *p) {
        register struct cmsghdr *cm = mtod(control, struct cmsghdr *);
	register tokenid_t *tokenid;
	struct pagstr *p_pag;
	struct ktokenstr *p_token;
	int i;

	KTDEBUG(5, "ktoken_unp_externalize(): called for proc %d\n", p->p_pid);

	if (cm->cmsg_type != SCM_KTOKENS) {
		KTDEBUG(1, "ktoken.ktoken_unp_externalize(): called on "
			"non-SCM_KTOKENS");
		return(EINVAL);
	}

	/* this is really a sanity check */
	if ((cm->cmsg_len - sizeof(*cm)) < sizeof(tokenid_t)) {
	    KTDEBUG(1, "ktoken.ktoken_unp_externalize(): cmsg too small\n");
	    return(EMSGSIZE);
	}

	tokenid = (tokenid_t *) (cm + 1);

	/* find the token by tokenid */
	
	t_internal_find_token_by_tokenid(*tokenid, &p_token);
	if (!p_token) {
	    /* this is a bad situation -- apparently the token created
	       in unp_internalize has disappeared (?).  This might
	       happen if the module were unloaded and reloaded between
	       the sendmsg and recvmsg, so we will just return an
	       error and hope for the best */
	    KTDEBUG(0, "ktoken.ktoken_unp_externalize(): transfered token "
		    "%lu not found\n", *tokenid);
	    *tokenid = 0; /* set the received token as invalid */
	    /* retrieve recipient PAG */
	    return(T_TOKENNOTFOUND);
	}

	t_internal_find_pag_by_proc(&p_pag, p);
	if (!p_pag) {
	    KTDEBUG(5, "ktoken.ktoken_unp_externalize(): proc has no pag\n");
	    /* refcount in already 1 from unp_internalize creation --
	       this results in garbage collection */
	    t_internal_decref_tokenstr(p_token);
	    return(T_PAGNOTFOUND);
	}

	/* add token ref to PAG */
	i = t_internal_add_token_to_pag(p_token, p_pag);
	if (i) {
            KTDEBUG(1, "ktoken.ktoken_unp_externalize(): could not add token "
		"to pag\n");
	    t_internal_decref_tokenstr(p_token);
	    return(i);
	}

	/* dec ref count because transfered copy has completed trip */
	t_internal_decref_tokenstr(p_token);
	
	return(0);
}

int ktoken_unp_internalize(struct mbuf *control, struct proc *p) {
        register struct cmsghdr *cm = mtod(control, struct cmsghdr *);
	register tokenid_t *tokenid;
	struct pagstr *p_pag;
	struct ktokenstr *p_token, *newtoken;
	int i;

	KTDEBUG(5, "ktoken_unp_internalize(): called for proc %d\n", p->p_pid);

	if (cm->cmsg_type != SCM_KTOKENS) {
		KTDEBUG(1, "ktoken.ktoken_unp_internalize(): called on "
			"non-SCM_KTOKENS");
		return(EINVAL);
	}

	if ((cm->cmsg_len - sizeof(*cm)) < sizeof(tokenid_t)) {
	    KTDEBUG(1, "ktoken.ktoken_unp_internalize(): cmsg too small\n");
	    return(EMSGSIZE);
	}

	tokenid = (tokenid_t *) (cm + 1);

	/* retrive sender PAG */
	t_internal_find_pag_by_proc(&p_pag, p);
	if (!p_pag) {
	    KTDEBUG(5, "ktoken.ktoken_unp_internalize(): proc has no pag\n");
	    return (T_PAGNOTFOUND);
	}
	/* see if they have the token */

	t_internal_find_token_by_tokenid(*tokenid, &p_token);
	if (!p_token) {
	    KTDEBUG(5, "ktoken.ktoken_unp_internalize(): token not found\n");
	    return(T_TOKENNOTFOUND);
	}

	KTDEBUG(5, "ktoken.ktoken_unp_internalize(): found requested token "
		"%lu\n", *tokenid);
	
	/* check they are allowed to send */
	if (!(KTOKEN_PERMITS(p_token, T_R_TRANSFERABLE)) &&
	    (KTOKEN_PERMITS(p_token, T_R_TRANSFERABLEONCE))) {
	    KTDEBUG(1, "ktoken.ktoken_unp_internalize(): token may not be "
		    "transferred\n");
	    return(EPERM);
	}

	/* make a copy of the token */

	i = t_internal_create_token(p_token->utoken.creator,
				    (struct pagstr *) 0, /* because kern */
				    &(p_token->utoken),
				    &newtoken);

	if (i) {
		KTDEBUG(1, "ktoken.ktoken_unp_internalize(): unable to create "
			"token\n");
		return(i);
	}

	/* adjust new token rights */
	newtoken->utoken.rights |= T_R_DELETE;             /* add delete */
	newtoken->utoken.rights &= ~T_R_TRANSFERABLEONCE;  /* remove t1 */
	    
	newtoken->refcount++;  /* increment refcount just because */

	/* store value as appropriate in cmsg structure */

	*tokenid = newtoken->utoken.tokenid;

	return(0);
}

int
ktoken_unp_gc(struct cmsghdr *control) {
	KTDEBUG(5, "ktoken.ktoken_unp_gc(): called\n");
	return(0);
}
#endif

