/*-
 * Copyright (c) 1998 Robert N. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name Robert N. Watson may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: ktoken_support.c,v 1.8 1998/07/06 00:18:41 robert Exp $
 */

/*
 * token support routines -- internal structure manipulation, etc.
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/exec.h>
#include <sys/sysent.h>
#include <sys/lkm.h>
#include <sys/malloc.h>
#include <sys/errno.h>
#include <sys/proc.h>
#include <sys/libkern.h>

#include "../common/ktoken_const.h"
#include "../common/ktoken_types.h"
#include "../common/ktoken_structs.h"
#include "../common/ktoken_syscall.h"
#include "ktoken.h"
#include "ktoken_kstructs.h"
#include "ktoken_support.h"
#include "ktoken_kconst.h"
#include "procmap.h"


/* malloc accounting types */
#ifdef MALLOC_DEFINE
MALLOC_DEFINE(M_TOKEN, "token", "authentication/authorization token storage");
MALLOC_DEFINE(M_TOKENREF, "tokenref", "token storage reference from pag");
MALLOC_DEFINE(M_PAG, "pag", "process authentication group storage");
#else
#define M_TOKEN M_SECA
#define M_TOKENREF M_SECA
#define M_PAG M_SECA
#endif


/* all tokens hashed by tokenid */
LIST_HEAD(tokhashhead, ktokenstr) *tokhashtbl;
u_long tokhash; /* size of hash table - 1 */

#define TOKHASH(tokenid) (&tokhashtbl[(tokenid) & tokhash])

/* other globals */
struct ktokenstr tok_dummy;	/* just for now to make dereferences work */
struct pagstr pag_dummy;	/* "" "" */
pagid_t maxpagid=1;		/* highest allocated so far */
tokenid_t maxtokenid=1;         /* "" "" */


/* init tables when module is loaded */
void t_internal_init(void) {
    tokhashtbl = hashinit(MAX_TOKENS / 4, M_TOKEN, &tokhash);
	pagmap_init();
}

/* remove table when module is removed */
void t_internal_shutdown(void) {
    struct pagstr *p_tmp;
	struct proc p;
    int i;

    for (i=0; i<PAGMAP_SIZE; i++) {
	p.p_pid = pag_mapping[i].pid;
	p_tmp = pag_mapping[i].pagstr;
	if (p.p_pid != 0) {
		KTDEBUG(5, "ktoken.t_internal_shutdown(): freeing proc %lu\n",
			p.p_pid);
		t_internal_remove_proc_pag(&p, p_tmp);
		t_internal_decref_pagstr(p_tmp);
		pag_mapping[i].pid = 0;
		pag_mapping[i].pagstr = 0;
	}
    }

    FREE(tokhashtbl, M_TOKEN);
}

int t_internal_token_has_expired(struct ktokenstr *p_token) {
    /* XXXXX compare expiry time with the current time */
    KTDEBUG(1, "ktoken.t_internal_token_has_expired(): not yet implemented\n");
    return(0);
}

int t_internal_token_is_tokend(struct ktokenstr *p_token) {
    int i;

    i =  (  /* creator */
	((p_token->utoken.creator == TOKEN_CRT_KERNEL) ||
	 (p_token->utoken.creator == TOKEN_CRT_TOKEND))
	&& /* major type */
	(p_token->utoken.major == TOKEN_MAJ_LOCALPRIV)
	&& /* minor type */
	(p_token->utoken.minor == TOKEN_LPRIV_TOKEND)
	);

    if (i) {
	KTDEBUG(5, "t_internal_token_is_tokend(): tokenid %lu is a TOKEND\n",
		p_token->utoken.tokenid);
    } else {
	KTDEBUG(5, "t_internal_token_is_tokend(): tokenid %lu is not a "
		"TOKEND\n", p_token->utoken.tokenid);
    }

    return(i);
}

/* return true if created by a good role, and has localpriv type */
int t_internal_token_is_real_localpriv(struct ktokenstr *p_token) {
    if ((p_token->utoken.creator != TOKEN_CRT_KERNEL) &&
	(p_token->utoken.creator != TOKEN_CRT_TOKEND)) {
	KTDEBUG(5, "ktoken.t_internal_token_is_real_localpriv(): bad "
		"creator\n");
	return(0);
    }

    if (p_token->utoken.major != TOKEN_MAJ_LOCALPRIV) {
	KTDEBUG(5, "ktoken.t_internal_token_is_real_localpriv(): bad "
		"major\n");
	return(0);
    }

    return(1);
}
							     
/* decrement refcount on a tokenstr, GC if necessary */
void t_internal_decref_tokenstr(struct ktokenstr *p_token) {
    if (!p_token) {
	KTDEBUG(0, "ktoken.t_internal_decref_tokenstr(): null p_token\n");
	return;
    }
    if (!p_token->refcount) {
	KTDEBUG(0, "ktoken.t_internal_decref_tokenstr(): refcount "
		"already 0 for tokenid %lu\n", p_token->utoken.tokenid);
	return;
    }
    if (p_token->refcount < 0) {
	KTDEBUG(0, "ktoken.t_internal_decref_tokenstr(): recount negative "
		"for tokenid %lu!\n", p_token->utoken.tokenid);
	return;
    }
    p_token->refcount--;
    if (!p_token->refcount) {
	KTDEBUG(1, "ktoken.t_internal_decref_tokenstr(): garbage collecting "
		"%lu\n", p_token->utoken.tokenid);
	LIST_REMOVE(p_token, t_hash);
	FREE(p_token, M_TOKEN);
    }
}

/* Both pagstr and tokenrefstr use refcounts to manage garbage collection. */
void t_internal_decref_tokenrefstr(struct ktokenrefstr *p_tokenref) {
    if (!p_tokenref) {
	KTDEBUG(0, "ktoken.t_internal_decref_tokenrefstr(): null "
		"p_tokenref\n");
	return;
    }
    if (!p_tokenref->refcount) {
	/* no references, so why dec 'em? */
	KTDEBUG(0, "ktoken.t_internal_decref_tokenrefstr(): "
		"refcount already 0\n");
	return;
    }
    
    KTDEBUG(5, "ktoken.t_internal_decref_tokenrefstr(): lowering refcount "
	    "ref to %lu\n", p_tokenref->token->utoken.tokenid);
    
    p_tokenref->refcount--;
    if (!p_tokenref->refcount) {
	KTDEBUG(1, "ktoken.t_internal_decref_tokenrefstr(): garbage "
		"collecting ref to %lu\n", p_tokenref->token->utoken.tokenid);

	/* this tokenrefstr is done for, so check its token */
	t_internal_decref_tokenstr(p_tokenref->token);
	LIST_REMOVE(p_tokenref, t_list);
	FREE(p_tokenref, M_TOKENREF);
    }
}

/* given a pag, walk its list of tokens and decrement all of their
   ref counts, remove from list, etc */
void t_internal_removetokenlist(struct pagstr *p_pagstr) {
    struct ktokenrefstr *tr;
    
    while (p_pagstr->t_list.lh_first != NULL) {
	tr = p_pagstr->t_list.lh_first;
	t_internal_decref_tokenrefstr(tr);
    }
}

/* decrement refcount on a PAG */
void t_internal_decref_pagstr(struct pagstr *p_pagstr) {
    if (!p_pagstr) {
	KTDEBUG(0, "ktoken.t_internal_decref_pagstr(): null p_pagstr\n");
	return;
    }
    if (!p_pagstr->refcount) {
	KTDEBUG(0, "ktoken.t_internal_decref_pagstr(): refcount already 0\n");
	return;
    }
    KTDEBUG(5, "ktoken.t_internal_decref_pagstr(): lowering refcount on %lu\n",
	    p_pagstr->pagid);
    
    p_pagstr->refcount--;

    if (!p_pagstr->refcount) {
	/* gc the pagstr as now has 0-refcount */
	KTDEBUG(1, "ktoken.t_internal_decref_pagstr(): garbage collection "
		"%lu\n", p_pagstr->pagid);
	t_internal_removetokenlist(p_pagstr);
	FREE(p_pagstr, M_PAG);
    }
}

/* return the pagid for a new PAG */
pagid_t t_internal_nextpagid(void) {
    return maxpagid++;
}

/*
 * Initialize a pag structure
 * Malloc pag variable given passed malloc options (fail if needed),
 * if successful, init fields as appropriate (using pagid, for example)
 */
void t_internal_newpag(struct pagstr **pp_pag,
			int flags) {
    struct pagstr *newpag;
    
    KTDEBUG(1, "ktoken.t_internal_newpag(): creating new pag\n");
    MALLOC(newpag, struct pagstr *, sizeof(struct pagstr), M_PAG, flags);
    if (!newpag) {
	/* user must have set MNOWAIT, so we got back null */
	KTDEBUG(1, "ktoken.t_internal_newpag(): MALLOC failed\n");
	*pp_pag = 0;
	return;
    }

    newpag->pagid = t_internal_nextpagid();
    newpag->refcount = 0;
    newpag->numtokens = 0;
    newpag->maxtokens = MAX_TOKENS_PAG; /* XXXXX */
    newpag->tokend_count = 0;

    LIST_INIT(&(newpag->t_list));
    *pp_pag = newpag;
    KTDEBUG(1, "ktoken.t_internal_newpag(): pag allocated pagid %lu\n",
            newpag->pagid);
}

/*
 * Copy appropriate token references from old to new -- don't use a non-
 * empty pag as new, as this doesn't check for duplicates!!!
 */
void t_internal_inheritpag(struct pagstr *old,
			   struct pagstr *new) {
    struct ktokenrefstr *ktr;
    int i;

    if (!old) {
	/* if old is null, do nothing */
	KTDEBUG(5, "ktoken.t_internal_inheritpag(): from null to %lu\n",
		new->pagid);
	return;
    } else {
	KTDEBUG(1, "ktoken.t_internal_inheritpag(): from %lu to %lu\n",
		old->pagid, new->pagid);

	/* walk list of tokens in old PAG */
	for (ktr = old->t_list.lh_first;
	     ktr != NULL;
	     ktr = ktr->t_list.le_next) {

	    /* if token is inheritable */
	    if (ktr->token->utoken.rights & T_R_INHERIT) {
		/* add token */
		i = t_internal_add_token_to_pag(ktr->token, new);
		if (i) {
		    KTDEBUG(0, "ktoken.t_internal_inheritpag(): error %d "
			    "when adding token %lu\n", i, 
			    ktr->token->utoken.tokenid);
		}
	    }
	}
	/* done iterating */
    }
}

/* remove a process from a PAG */
void t_internal_remove_proc_pag(struct proc *p, struct pagstr *old) {
    KTDEBUG(1, "ktoken.t_internal_remove_proc_pag(): remove pid %d from pag "
	    "%lu\n", p->p_pid, old->pagid);

    pagmap_remove_map(p); /* proc hash table hack */
    t_internal_decref_pagstr(old);
}

/* add a process to a PAG */
void t_internal_add_proc_pag(struct proc *p, struct pagstr *new) {
    KTDEBUG(1, "ktoken.t_internal_add_proc_pag(): add pid %d to pag %lu\n",
	    p->p_pid, new->pagid);
    new->refcount++;
    pagmap_add_map(p, new); /* proc hash table hack */
}

/* find the PAG a process is in */
void t_internal_find_pag_by_proc(struct pagstr **pag, struct proc *p) {
    *pag = 0;
    *pag = pagmap_get_map(p); /* proc hash table hack */
}

/* given a tokenid, return the token */
void t_internal_find_token_by_tokenid(tokenid_t tokenid,
				      struct ktokenstr **token) {
    struct ktokenstr *kt;
    
    KTDEBUG(5, "ktoken.t_internal_find_token_by_tokenid(): search for %lu\n",
	    tokenid);
    
    /* iterate on hash chain of tokenid until it is found */
    for (kt = TOKHASH(tokenid)->lh_first; kt != 0; kt = kt->t_hash.le_next)
	if (kt->utoken.tokenid == tokenid) {

	    /* found it */
	    *token = kt;
	    KTDEBUG(5, "ktoken.t_internal_find_token_by_tokenid(): "
		    "found %lu\n", tokenid);
            KTDEBUG(10, "ktoken.t_internal_find_token_by_tokenid(): "
                    "token %lu has types (%d,%d,%d)\n", kt->utoken.tokenid,
                    kt->utoken.major, kt->utoken.minor,
                    kt->utoken.minorminor);
	    return;
	}

    /* didn't find it */
    KTDEBUG(5, "ktoken.t_internal_find_token_by_tokenid(): didn't find %lu\n",
	    tokenid);
    *token = 0;
}

/* give a pag, determine if it has a valid local priv token TOKEND */
int t_internal_pag_has_tokend(struct pagstr *p_pag) {
    if (!(p_pag->tokend_count)) {
	KTDEBUG(5, "ktoken.t_internal_pag_has_tokend(): pag %lu does not "
		"have TOKEND\n", p_pag->pagid);
	return 0;
    }

    /* XXXXX iterate on tokens looking at each tokend token to see if it
       has expired.  If an unexpired token is found, return true, else
       return false */

    KTDEBUG(1, "ktoken.t_internal_pag_has_tokend(): under-implemented\n");
    return(1);
}

/* given a token and a PAG, find the PAG's reference to the token (if there
   is one) */
void t_internal_pag_get_tokenref(struct pagstr *p_pag,
				 struct ktokenstr *p_token,
				 struct ktokenrefstr **pp_tokenref) {
    struct ktokenrefstr *ktr;

    KTDEBUG(5, "ktoken.t_internal_pag_get_tokenref(): search for token %lu "
	    "in pag %lu\n", p_token->utoken.tokenid, p_pag->pagid);
    
    /* iterate token ref list in PAG */
    for (ktr = p_pag->t_list.lh_first; ktr != 0; ktr = ktr->t_list.le_next) {
	if (ktr->token == p_token) {
	    /* found one with the same token */
	    *pp_tokenref = ktr;
	    return;
	}
    }

    /* didn't find it */
    *pp_tokenref = 0;
}

void t_internal_copy_token_fields(struct utokenstr *dest,
				  struct utokenstr *source,
				  u_int fields) {
    /* copy fields indicated by fields to dest from source */
    
    if (fields & TOKEN_FIELD_TOKENID)
	dest->tokenid = source->tokenid;

    if (fields & TOKEN_FIELD_MAJOR)
	dest->major = source->major;

    if (fields & TOKEN_FIELD_MINOR)
	dest->minor = source->minor;

    if (fields & TOKEN_FIELD_MINORMINOR)
	dest->minorminor = source->minorminor;

    if (fields & TOKEN_FIELD_NAME)
	memcpy(dest->name, source->name, T_STRINGLEN_MAX);
   
    if (fields & TOKEN_FIELD_INSTANCE)
	memcpy(dest->instance, source->instance, T_STRINGLEN_MAX);

    if (fields & TOKEN_FIELD_REALM)
	memcpy(dest->realm, source->realm, T_STRINGLEN_MAX);

    if (fields & TOKEN_FIELD_RIGHTS)
	dest->rights = source->rights;

    if (fields & TOKEN_FIELD_PUBLICDAT) {
	dest->publiclen = source->publiclen;
	memcpy(dest->publicdat, source->publicdat, T_DATALEN_MAX);
    }

    if (fields & TOKEN_FIELD_PRIVATEDAT) {
	dest->privatelen = source->privatelen;
	memcpy(dest->privatedat, source->privatedat, T_DATALEN_MAX);
    }

    if (fields & TOKEN_FIELD_EXPIRETIME)
	dest->expiretime = source->expiretime;
    
    if (fields & TOKEN_FIELD_CREATETIME)
	dest->createtime = source->createtime;

    if (fields & TOKEN_FIELD_CREATOR)
	dest->creator = source->creator;
}

int t_internal_token_matches(u_int whichfields,
			     struct utokenstr *p_token1,
			     struct utokenstr *p_token2) {
    KTDEBUG(5, "ktoken.t_internal_token_matches(): comparing on fields "
	    "%du\n", whichfields);

    /* go down the list of fields, return 0 on first match failure */
    if (whichfields & TOKEN_FIELD_TOKENID) {
	if (p_token1->tokenid != p_token2->tokenid) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): tokenid match "
		    "failed\n");
	    return 0;
	}
    }

    if (whichfields & TOKEN_FIELD_NAME) {
	if (strncmp(p_token1->name, p_token2->name,
		    T_STRINGLEN_MAX)) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): name match "
		    "failed\n");
	    return 0;
	}
    }

    if (whichfields & TOKEN_FIELD_REALM) {
	if (strncmp(p_token1->realm, p_token2->realm,
		    T_STRINGLEN_MAX)) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): realm match "
		    "failed\n");
	    return 0;
	}
    }

    if (whichfields & TOKEN_FIELD_RIGHTS) {
	if (p_token1->rights != p_token2->rights) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): rights match "
		    "failed\n");
	    return 0;
	}
    }

    if (whichfields & TOKEN_FIELD_MAJOR) {
	if (p_token1->major != p_token2->major) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): major match "
		    "failed\n");
	    return 0;
	} 
    }

    if (whichfields & TOKEN_FIELD_MINOR) {
	if (p_token1->minor != p_token2->minor) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): minor match "
		    "failed\n");
	    return 0;
	}
    }

    if (whichfields & TOKEN_FIELD_MINORMINOR) {
	if (p_token1->minorminor != p_token2->minorminor) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): minorminor match "
		    "failed\n");
	    return 0;
	}
    }

    if (whichfields & TOKEN_FIELD_PUBLICDAT) {
	/* complex -- check has rights first, fail if not */
	if (!(p_token1->rights & T_R_READ)) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): publicdat match "
		    "failed due to token1 rights\n");
	    return 0;
	}
	if (!(p_token2->rights & T_R_READ)) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): publicdat match "
		    "failed due to token2 rights\n");
	    return 0;
	}
	/* both have rights to read, so do a real compare */
	if (p_token1->publiclen != p_token2->publiclen) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): publicdat match "
		    "failed\n");
	    return 0;
	}
	if (memcmp(p_token1->publicdat, p_token2->publicdat,
		   p_token1->publiclen)) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): publicdat match "
		    "failed\n");
	    return 0;
	}
    }

    if (whichfields & TOKEN_FIELD_PRIVATEDAT) {
	/* complex -- check has rights first, fail if not */
	if (!(p_token1->rights & T_R_READ)) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): privatedat match "
		    "failed due to token1 rights\n");
	    return 0;
	}
	if (!(p_token2->rights & T_R_READ)) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): privatedat match "
		    "failed due to token2 rights\n");
	    return 0;
	}
	/* both have rights to read, so do a real compare */
	if (p_token1->privatelen != p_token2->privatelen) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): privatedat match "
		    "failed\n");
	    return 0;
	}
	if (memcmp(p_token1->privatedat, p_token2->privatedat,
		   p_token1->privatelen)) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): privatedat match "
		    "failed\n");
	    return 0;
	}
    }

    if (whichfields & TOKEN_FIELD_EXPIRETIME) {
	if (p_token1->expiretime != p_token2->expiretime) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): expiretime match "
		    "failed\n");
	    return 0;
	}
    }

    if (whichfields & TOKEN_FIELD_CREATETIME) {
	if (p_token1->createtime != p_token2->createtime) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): createtime match "
		    "failed\n");
	    return 0;
	}
    }

    if (whichfields & TOKEN_FIELD_CREATOR) {
	if (p_token1->creator != p_token2->creator) {
	    KTDEBUG(5, "ktoken.t_internal_token_matches(): creator match "
		    "failed\n");
	    return 0;
	}
    }

    KTDEBUG(5, "ktoken.t_internal_token_matches(): match succeeded\n");
    return 1;
}


/* given a pag and token, return an appropriate version for the process */
int t_internal_read_token(struct pagstr *p_pag,
			  struct ktokenstr *p_token,
			  struct utokenstr *outtoken) {
    struct ktokenrefstr *ktref; /* not used */
    struct utokenstr *ut; /* shortcut */
    u_int fields;

    KTDEBUG(5, "ktoken.t_internal_read_token(): called on token %lu by "
	    "pag %lu\n", p_token->utoken.tokenid, p_pag->pagid);
   
    /* first blank the whole thing to prevent data leaks */
    bzero(outtoken, sizeof(struct utokenstr));
    
    /* check to see if this pag can read this token */
    t_internal_pag_get_tokenref(p_pag, p_token, &ktref);
    if (ktref) {
	/* can access the token, now check rights */
	ut = &(p_token->utoken);
	KTDEBUG(10, "ktoken.t_internal_read_token(): token %lu has types: "
		"(%d,%d,%d)\n", ut->tokenid, ut->major, ut->minor,
		ut->minorminor);

	/* see which fields to copy by checking rights */
	fields = TOKEN_FIELD_ALL & ((~TOKEN_FIELD_PUBLICDAT) &
				    (~TOKEN_FIELD_PRIVATEDAT));
	if (KTOKEN_PERMITS(p_token, T_R_READ)) {
	    /* is readable */
	    KTDEBUG(5, "ktoken.t_internal_read_token(): token %lu has READ "
		    "permission set\n", p_token->utoken.tokenid);
	    fields |= TOKEN_FIELD_PUBLICDAT | TOKEN_FIELD_PRIVATEDAT;
	} else {
	    /* is not readable */
	    KTDEBUG(5, "ktoken.t_internal_read_token() token %lu does not "
		    "have READ permission set\n", p_token->utoken.tokenid);
	}

	/* copy the fields */
	t_internal_copy_token_fields(outtoken, ut, fields);
	return(0);
    } else {
	/* XXXX -- what should error be if pag doesn't reference the token? */
	KTDEBUG(1, "ktoken.t_internal_read_token(): pag %lu cannot access "
		"token %lu\n", p_pag->pagid, p_token->utoken.tokenid);
	return(EPERM);
    }
}

/* given a mintoken, a model token, and a set of fields to compare, find
 * the next token after mintokenid that matches */
void t_internal_search_token(struct pagstr *p_pag,
			     tokenid_t mintokenid,
			     u_int whichfields,
			     struct utokenstr *model,
			     struct ktokenstr **p_token) {
    struct ktokenrefstr *ktr;

    KTDEBUG(5, "ktoken.t_internal_search_token(): pag %lu requests search "
	    "with mintokenid %lu, %du fields\n", p_pag->pagid, mintokenid,
	    whichfields);

    /* iterate across the PAG's token list */
    for (ktr = p_pag->t_list.lh_first; ktr != 0; ktr = ktr->t_list.le_next) {
	/* is it sufficiently high tokenid? */
	if (ktr->token->utoken.tokenid >= mintokenid) {
	    /* does it have matching fields? */
	    if (t_internal_token_matches(whichfields, &(ktr->token->utoken),
					 model)) {
		/* if so, return it */
		*p_token = ktr->token;
		return;
	    }
	}
    }

    /* not found */
    *p_token = 0;
}

/* modify a token based on origin, a PAG to find the original, the token
 * to modify, a model token, and which fields to set */
int t_internal_modify_token(int origin,
			    struct pagstr *p_pag,
			    struct ktokenstr *p_token,
			    struct utokenstr *model,
			    int fields) {
    struct ktokenrefstr *temp;
    int testmajor, testminor, testminorminor;

    /* check rights on token */
    KTDEBUG(5, "ktoken.t_internal_modify_token(): mod token %lu by ident %d\n",
	    p_token->utoken.tokenid, origin);
    if (origin != TOKEN_CRT_KERNEL) {
	KTDEBUG(5, "ktoken.t_internal_modify_token(): mod by pag %lu\n",
		p_pag->pagid);
    }

    if (!(p_token->utoken.rights & T_R_MODIFY) &&
	(origin != TOKEN_CRT_KERNEL)) {
	KTDEBUG(2, "ktoken.t_internal_modify_token(): permission to modify "
		"token %lu denied\n", p_token->utoken.tokenid);
	return(EPERM);
    }

    /* prevent changes to bad choices of fields */
    if ((fields & TOKEN_FIELD_TOKENID) || 
	(fields & TOKEN_FIELD_CREATOR)) {
	KTDEBUG(1, "ktoken.t_internal_modify_token(): attempt to modify "
		"tokenid or creator rejected\n");
	return(EPERM);
    }

    if (p_pag) {
	/* user-space requests may only modify token in their pag*/
	t_internal_pag_get_tokenref(p_pag, p_token, &temp);
	if (!temp) {
	    KTDEBUG(2, "ktoken.t_internal_modify_token(): permission to "
		    "modify token %lu denied to pag %lu\n",
		    p_token->utoken.tokenid, p_pag->pagid);
	    return(EPERM);
	}
    } else {
	/* no pag */
	if (origin != TOKEN_CRT_KERNEL) {
	    KTDEBUG(2, "ktoken.t_internal_modify_token(): permission to "
		    "modify denied as no pag used\n");
	    return(EPERM);
	}
    }

    /* check that the pag could result in the creation of such a
       token via ktoken_createtoken() */

    if (fields & TOKEN_FIELD_MAJOR)
	testmajor = model->major;
    else testmajor = p_token->utoken.major;

    if (fields & TOKEN_FIELD_MINOR)
	testminor = model->minor;
    else testminor = p_token->utoken.minor;

    if (fields & TOKEN_FIELD_MINORMINOR)
	testminorminor = model->minor;
    else testminorminor = p_token->utoken.minorminor;

    /* check */
    if (!(t_internal_may_create_token_type(origin,
					   p_pag,
					   testmajor,
					   testminor,
					   testminorminor))) {
	KTDEBUG(1, "ktoken.t_internal_create_token(): invalid creation "
		"type (%d,%d,%d)\n", testmajor, testmajor, testminorminor);
	return(EPERM);	
    }

    /* either is from kernel, or appropriate pag and rights */
    /* update origin field */

    p_token->utoken.creator = origin;

    /* now walk the fields and act appropriately */

    t_internal_copy_token_fields(&(p_token->utoken), model, fields);

    KTDEBUG(5, "ktoken.t_internal_modify_token(): modify of tokenid %lu "
	    "succeeded\n", p_token->utoken.tokenid);

    return(0);
}

/* Given a token, pag, add a reference to that token in the pag.
   Assumes that the token is not already present in the pag */
int t_internal_add_token_to_pag(struct ktokenstr *p_token,
				 struct pagstr *p_pag) {
    struct ktokenrefstr *ktr, *temp, *prev;

    KTDEBUG(5, "ktoken.t_internal_add_token_to_pag(): add token %lu to "
	    "pag %lu\n", p_token->utoken.tokenid, p_pag->pagid);

    /* enforce resource limit */
    if (p_pag->numtokens + 1 > p_pag->maxtokens) {
	KTDEBUG(1, "ktoken.t_internal_add_token_to_pag(): pag %lu has "
		"maxtokens in add of %lu\n", p_token->utoken.tokenid);
	return(EAGAIN);
    }

    /* space for new reference */
    MALLOC(ktr, struct ktokenrefstr *, sizeof(struct ktokenrefstr), M_TOKENREF,
	   M_WAITOK);

    /* init reference */
    ktr->refcount = 1;
    ktr->token = p_token;
    
    /* update PAG */
    p_token->refcount++;
    p_pag->numtokens++;

    /* TOKEND */
    if (t_internal_token_is_tokend(p_token)) {
	/* is a local privilege token, up the count */
	p_pag->tokend_count++;
    }
	

    /* order tokens, so have to find a good spot */
    /* find first token higher than it, then insert */
    if (!p_pag->t_list.lh_first) {
	/* list currently empty */
	KTDEBUG(10, "ktoken.t_internal_add_token_to_pag(): new token list (tokenid %lu)\n", ktr->token->utoken.tokenid);
	LIST_INSERT_HEAD(&(p_pag->t_list), ktr, t_list);
	return(0);
    }
    /* list has stuff in it, find a good place and insert */
    /* XXXX damn slow */
    for (temp = p_pag->t_list.lh_first;
	 temp != 0;
	 temp = temp->t_list.le_next) {
	if (ktr->token->utoken.tokenid > p_token->utoken.tokenid) {
	    /* this is a good place to insert before */
	    KTDEBUG(10, "ktoken.t_internal_add_token_to_pag(): inserting "
		    "tokenid %lu before %lu\n", ktr->token->utoken.tokenid,
		    temp->token->utoken.tokenid);
	    LIST_INSERT_BEFORE(temp, ktr, t_list); 
	    return(0);
	}
	prev = temp;
    }

    /* fell through, nothing higher */
    KTDEBUG(10, "ktoken.t_internal_add_token_to_pag(): appending tokenid %lu "
	    "to list after %lu\n", ktr->token->utoken.tokenid,
	    prev->token->utoken.tokenid);
    LIST_INSERT_AFTER(prev, ktr, t_list);
    return(0);
}

/* what tokenid should the next created token have? */
tokenid_t t_internal_nexttokenid(void) {
    return maxtokenid++;
}

/* access control point for creation of certain types of tokens */
int t_internal_may_create_token_type(int origin,
				     struct pagstr *pag,
				     int major,
				     int minor,
				     int minorminor) {
    if ((origin != TOKEN_CRT_KERNEL) &&
	((minor < 0) || (major < 0) || (minorminor < 0))) {
	/* only the kernel may create reflection tokens */
	return(0);
    }

    /* should check origin TOKEN_CRT_TOKEND and pag for agreement */
    return(1);

}

/* origin - USER, KERN
 * pag -- the pag requesting creation
 * model -- the model
 * tokenid -- return the tokenid here
 * returns an errno-style error number
 */
int t_internal_create_token(int origin,
			    struct pagstr *p_pag,
			    struct utokenstr *model,
			    struct ktokenstr **outtoken) {
    struct ktokenstr *kt;
    u_int fieldstocopy;

    if (p_pag) {
	KTDEBUG(5, "ktoken.t_internal_create_token(): request by origin %d, "
		"pag %lu to create token (%d,%d,%d)\n", origin, p_pag->pagid,
		model->major, model->minor, model->minorminor);
    } else {
	KTDEBUG(5, "ktoken.t_internal_create_token(): request by origin %d "
		"to create token (%d,%d,%d)\n", origin, model->major,
		model->minor, model->minorminor);
    }

    /* may the requestor create this type of token? */
    if (!(t_internal_may_create_token_type(origin,
					   p_pag,
					   model->major,
					   model->minor,
					   model->minorminor))) {
	KTDEBUG(1, "ktoken.t_internal_create_token(): token creation "
		"request failed from pag %lu\n", p_pag->pagid);
	return(EPERM);
    }

    /* check resource limit */
    if (p_pag->numtokens >= p_pag->maxtokens) {
	KTDEBUG(1, "ktoken.t_internal_create_token(): token creation failed "
		"due to rsrc limit in pag %lu\n", p_pag->pagid);
	return(EAGAIN);
    }

    /* get space for token */
    MALLOC(kt, struct ktokenstr *, sizeof(struct ktokenstr), M_TOKEN,
	   M_WAITOK);

    /* have token, initialize */
    bzero(kt, sizeof(struct ktokenstr));

    /* set tokenid */
    kt->utoken.tokenid = t_internal_nexttokenid();

    /* copy appropriate fields */
    fieldstocopy = TOKEN_FIELD_ALL & ((~TOKEN_FIELD_TOKENID) &
				      (~TOKEN_FIELD_CREATOR));
    t_internal_copy_token_fields(&(kt->utoken), model, fieldstocopy);

    /* set creator to request origin */
    kt->utoken.creator = origin;

    /* add to global hash */
    LIST_INSERT_HEAD(TOKHASH(kt->utoken.tokenid), kt, t_hash);
    kt->refcount = 0;
    *outtoken = kt;

    KTDEBUG(5, "ktoken.t_internal_create_token(): created token %lu with types"
		"(%d,%d,%d)\n", kt->utoken.tokenid, kt->utoken.major,
		kt->utoken.minor, kt->utoken.minorminor);

    return(0);
}

/* pag -- pag requesting the reflection
 * p_token -- the token to be reflected
 * tokenid -- return the new tokenid here
 */
int t_internal_reflect_token(struct pagstr *p_pag,
			     struct ktokenstr *p_token,
			     struct ktokenstr **outtoken) {
    struct ktokenstr *kt;
    u_int fieldstocopy;

    KTDEBUG(5, "ktoken.t_internal_reflect_token(): request by pag %lu to "
	    "reflect tokenid %lu\n", p_pag->pagid, p_token->utoken.tokenid);

    /* check this is a reflectable token */
    if ((p_token->utoken.major < 0) || (p_token->utoken.minor < 0) ||
	(p_token->utoken.minorminor < 0)) {
	KTDEBUG(5, "ktoken.t_internal_reflect_token(): request to reflect "
		"tokenid %lu failed due to types bad (%d,%d,%d)\n",
		p_token->utoken.tokenid, p_token->utoken.major,
		p_token->utoken.minor, p_token->utoken.minorminor);
	return(EPERM);
    }

    /* enforce resource limit */
    if (p_pag->numtokens >= p_pag->maxtokens) {
	KTDEBUG(1, "ktoken.t_internal_reflect_token(): token reflect failed "
		"due to rsrc limit in pag %lu\n", p_pag->pagid);
	return(EAGAIN);
    }

    /* malloc space for token */
    MALLOC(kt, struct ktokenstr *, sizeof(struct ktokenstr), M_TOKEN,
	   M_WAITOK);
    bzero(kt, sizeof(struct ktokenstr));

    /* give a fresh tokenid */
    kt->utoken.tokenid = t_internal_nexttokenid();

    /* what fields to copy */
    fieldstocopy = TOKEN_FIELD_ALL & ((~TOKEN_FIELD_TOKENID) &
				      (~TOKEN_FIELD_MAJOR) &
				      (~TOKEN_FIELD_MINOR) &
				      (~TOKEN_FIELD_MINORMINOR) &
				      (~TOKEN_FIELD_PUBLICDAT) &
				      (~TOKEN_FIELD_PRIVATEDAT));
    t_internal_copy_token_fields(&(kt->utoken), &(p_token->utoken),
				 fieldstocopy);

    /* negative types because a reflection */
    kt->utoken.major = -1 * (p_token->utoken.major);
    kt->utoken.minor = -1 * (p_token->utoken.minor);
    kt->utoken.minorminor = -1 * (p_token->utoken.minorminor);

    /* bzero reset the dat fields already for us */

    /* add delete bit, remove modify bit */
    kt->utoken.rights |= T_R_DELETE;
    kt->utoken.rights &= ~T_R_MODIFY;

    /* add to global hash */
    LIST_INSERT_HEAD(TOKHASH(kt->utoken.tokenid), kt, t_hash);
    kt->refcount = 0;

    /* return a reference */
    *outtoken = kt;
    KTDEBUG(5, "ktoken.t_internal_reflect_token(): tokenid %lu a reflection "
	    "of tokenid %lu\n", kt->utoken.tokenid, p_token->utoken.tokenid);

    return(0);
}

/* remove a reference to a token from a PAG */
int t_internal_remove_token_from_pag(struct pagstr *p_pag,
				     struct ktokenstr *p_token) {
    struct ktokenrefstr *temp;
    tokenid_t tokenid;

    /* find the existing reference to remove */
    t_internal_pag_get_tokenref(p_pag, p_token, &temp);
    if (!temp) {
	/* there is no such reference */
	KTDEBUG(0, "ktoken.t_internal_remove_token_from_pag(): token %lu not "
		"in pag %lu\n", p_token->utoken.tokenid, p_pag->pagid);
	return(T_TOKENNOTFOUND);
    }

    tokenid = p_token->utoken.tokenid; /* can't print after delete */

    /* remove from PAG reference list */
    LIST_REMOVE(temp, t_list);
    p_pag->numtokens--;

    if (t_internal_token_is_tokend(p_token)) {
	/* is a local privilege token, up the count */
	p_pag->tokend_count--;
    }
    /* dec refcount on token ref */
    t_internal_decref_tokenrefstr(temp);
    
    KTDEBUG(5, "ktoken.t_internal_remove_token_from_pag(): success in removal "
	    "of tokenid %lu\n", tokenid);
    return(0);
}

/* given a request origin, pag, and token, delete pag reference */
int t_internal_delete_token(int origin,
			    struct pagstr *p_pag,
			    struct ktokenstr *p_token) {
    struct ktokenrefstr *temp;
    tokenid_t tokenid;
    int i;

    KTDEBUG(5, "ktoken.t_internal_delete_token(): del token %lu by ident %d\n",
	    p_token->utoken.tokenid, origin);
    if (origin != TOKEN_CRT_KERNEL) {
	KTDEBUG(5, "ktoken.t_internal_delete_token(): del by pag %lu\n",
		p_pag->pagid);
    }

    /* is token deletable? */
    if (!(p_token->utoken.rights & T_R_DELETE) &&
	(origin != TOKEN_CRT_KERNEL)) {
	KTDEBUG(2, "ktoken.t_internal_delete_token(): permission to delete "
		"token %lu denied\n", p_token->utoken.tokenid);
	return(EPERM);
    }

    tokenid = p_token->utoken.tokenid;

    /* go ahead and remove */
    i = t_internal_remove_token_from_pag(p_pag, p_token);
    if (i) return(i);
    
    KTDEBUG(5, "ktoken.t_internal_delete_token(): delete of tokenid %lu "
	    "succeeded\n", tokenid);

    return (0);
}
