/*-
 * Copyright (c) 1998 Robert N. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name Robert N. Watson may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: ktoken.c,v 1.4 1998/06/26 13:17:34 robert Exp $
 */

/*
 * ktoken syscall implementation, kernel event hooks (fork, exit)
 */


#include <sys/param.h>
#include <sys/systm.h>
#include <sys/exec.h>
#include <sys/sysent.h>
#include <sys/lkm.h>
#include <sys/malloc.h>
#include <sys/errno.h>
#include <sys/proc.h>

#include "../common/ktoken_const.h"
#include "../common/ktoken_types.h"
#include "../common/ktoken_structs.h"
#include "../common/ktoken_syscall.h"
#include "ktoken_kconst.h"
#include "ktoken_kstructs.h"
#include "ktoken.h"
#include "ktoken_support.h"

extern int ktoken __P((struct proc *p, struct t_syscall_args *uap));

/* globals */
int	ktdebug;
static pagid_t		badpagid=0;


/* syscall multiplex -- this is the real syscall */
int
ktoken(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    int fn = uap->whichfunction;
    KTDEBUG(2,"ktoken(): function %d\n", uap->whichfunction);
    
    /* check validity of argument */
    if (
	!((fn >= TOKEN_CONTROL_MIN) &&
	  (fn <= TOKEN_CONTROL_MAX))
	&&
	!((fn >= TOKEN_PAGCALL_MIN) &&
	  (fn <= TOKEN_PAGCALL_MAX))
	&&
	!((fn >= TOKEN_TOKCALL_MIN) &&
	  (fn <= TOKEN_TOKCALL_MAX))
	)
    {
	KTDEBUG(1,"ktoken(): invalid function call\n");
	return(EINVAL);
    }
    switch(fn) {
    case TOKEN_SYSCALL_DEBUG:
	KTDEBUG(2,"ktoken.ktoken_debug()\n");
	return ktoken_debug(p, uap);
    case TOKEN_SYSCALL_NEWPAG:
	KTDEBUG(2,"ktoken.ktoken_newpag()\n");
	return ktoken_newpag(p, uap);
    case TOKEN_SYSCALL_GETPAG:
	KTDEBUG(2,"ktoken.ktoken_getpag()\n");
	return ktoken_getpag(p, uap);
    case TOKEN_SYSCALL_READTOKEN:
	KTDEBUG(2,"ktoken.ktoken_readtoken()\n");
	return ktoken_readtoken(p, uap);
    case TOKEN_SYSCALL_MODIFYTOKEN:
	KTDEBUG(2,"ktoken.ktoken_modifytoken()\n");
	return ktoken_modifytoken(p, uap);
    case TOKEN_SYSCALL_CREATETOKEN:
	KTDEBUG(2,"ktoken.ktoken_createtoken()\n");
	return ktoken_createtoken(p, uap);
    case TOKEN_SYSCALL_REFLECTTOKEN:
	KTDEBUG(2,"ktoken.ktoken_reflecttoken()\n");
	return ktoken_reflecttoken(p, uap);
    case TOKEN_SYSCALL_DELETETOKEN:
	KTDEBUG(2,"ktoken.deletetoken()\n");
	return ktoken_deletetoken(p, uap);
    case TOKEN_SYSCALL_FINDTOKEN:
	KTDEBUG(2,"ktoken.findtoken()\n");
	return ktoken_findtoken(p, uap);
    default:
	KTDEBUG(0,"ktoken(): invalid function beyond check\n");
	return(EINVAL);
    }
    
    /* never get here */
    return(EINVAL);		/* success (or error code from errno.h)*/
}

/* macros to copy typed data from/into user space */
#define T_COPYIN(src, dest, type) copyin(src, dest, sizeof(type))
#define T_COPYOUT(src, dest, type) copyout(src, dest, sizeof(type))

/* make this process part of a new pag */
static int
ktoken_newpag(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_newpag_args a;
    int i;
    
    struct pagstr *new, *old;
    
    i = T_COPYIN(uap->args, &a, struct t_newpag_args);
    if (i) return(i);
    
    /* find the current pag */
    t_internal_find_pag_by_proc(&old, p);
    
    /* create a new pag */
    t_internal_newpag(&new, M_WAITOK);
    if (!new) {
	/* t_internal_return_pagstr(old); */
	KTDEBUG(1, "ktoken.ktoken_newpag(): t_internal_newpag() failed\n");
	return(EAGAIN);
    }
    
    /* if there was an old one, inherit its tokens, but leave it */
    if (old) {
	t_internal_inheritpag(old, new);
	t_internal_remove_proc_pag(p, old);
    }
    /* add the current proc to the new pag */
    t_internal_add_proc_pag(p, new);
    
    /* return new pagid */
    i = T_COPYOUT(&(new->pagid), a.p_pagid, pagid_t);
    return(i);
}

/* return the current process' pagid */
static int
ktoken_getpag(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_getpag_args a;
    struct pagstr *p_pag;
    int i;
    
    i = T_COPYIN(uap->args, &a, struct t_getpag_args);
    if (i) return(i);
    
    /* find the current pag */
    t_internal_find_pag_by_proc(&p_pag, p);
    
    if (!p_pag) {
	return(T_PAGNOTFOUND);
    }
    
    /* return the pagid */
    i = T_COPYOUT(&(p_pag->pagid), a.p_pagid, pagid_t);
    return(i);
}

/* return a copy of an identified token */
static int
ktoken_readtoken(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_readtoken_args a;
    struct pagstr *p_pag;
    struct ktokenstr *p_token;
    struct utokenstr outtoken;	/* is this too big for stack? */
    int i;
    
    i = T_COPYIN(uap->args, &a, struct t_readtoken_args);
    if (i) return i;
    
    /* find the current pag */
    t_internal_find_pag_by_proc(&p_pag, p);
    if (!p_pag) {
	return (T_PAGNOTFOUND);
    }
    
    /* find the appropriate token */
    t_internal_find_token_by_tokenid(a.tokenid, &p_token);
    if (!p_token) {
	return(T_TOKENNOTFOUND);
    }
    
    /* create an appropriate-looking copy of the tokens */
    i = t_internal_read_token(p_pag, p_token, &outtoken);
    
    if (i) {
	return(i);
    }
    
    /* return contents of token */
    i = T_COPYOUT(&outtoken, a.p_token, struct utokenstr);
    return(i);
}

/* change specific fields of a token */
static int
ktoken_modifytoken(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_modifytoken_args a;
    struct pagstr *p_pag;
    struct utokenstr modeltoken;
    struct ktokenstr *p_token;
    int i;
    
    i = T_COPYIN(uap->args, &a, struct t_modifytoken_args);
    if (i) return(i);
    
    /* the token field model */
    i = T_COPYIN(a.p_token, &modeltoken, struct utokenstr);
    if (i) return(i);
    
    /* find the current pag */
    t_internal_find_pag_by_proc(&p_pag, p);
    if (!p_pag) {
	return (T_PAGNOTFOUND);
    }
    
    /* find the appropriate destination token */
    t_internal_find_token_by_tokenid(a.tokenid, &p_token);
    if (!p_token) {
	/* t_internal_return_pagstr(p_pag); */
	return (T_TOKENNOTFOUND);
    }
    
    /* modify the token in the "process" role */
    i = t_internal_modify_token(TOKEN_CRT_PROCESS, p_pag, p_token,
				&modeltoken, a.whichfields);
    
    return(i);
}

/* create a new token from model */
static int 
ktoken_createtoken(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_createtoken_args a;
    struct pagstr *p_pag;
    struct utokenstr modeltoken; /* XXXX stack space? */
    struct ktokenstr *outtoken;
    int i;
    
    i = T_COPYIN(uap->args, &a, struct t_createtoken_args);
    if (i) return(i);
    
    /* token to create from */
    i = T_COPYIN(a.p_token, &modeltoken, struct utokenstr);
    if (i) return(i);
    
    /* find current PAG */
    t_internal_find_pag_by_proc(&p_pag, p);
    if (!p_pag) {
	return (T_PAGNOTFOUND);
    }
    
    /* create the token in "process" role */
    i = t_internal_create_token(TOKEN_CRT_PROCESS, p_pag, &modeltoken,
				&outtoken);
    if (i) return(i);
    
    /* add the token to the current PAG */
    i = t_internal_add_token_to_pag(outtoken, p_pag);
    if (i) {
	/* but if we can't, destroy the token */
	outtoken->refcount++; /* XXXX HACK to make gc'able */
	t_internal_decref_tokenstr(outtoken);
	return(i);
    }
    
    /* return tokenid */
    i = T_COPYOUT(&(outtoken->utoken.tokenid), a.p_tokenid, tokenid_t);
    return(i);
}

/* create a reflection of a token */
static int 
ktoken_reflecttoken(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_reflecttoken_args a;
    struct pagstr *p_pag;
    struct ktokenstr *p_token, *outtoken;
    int i;
    
    i = T_COPYIN(uap->args, &a, struct t_reflecttoken_args);
    if (i) return(i);
    
    /* find the current PAG */
    t_internal_find_pag_by_proc(&p_pag, p);
    if (!p_pag) {
	return (T_PAGNOTFOUND);
    }

    /* find the source token */
    t_internal_find_token_by_tokenid(a.tokenid1, &p_token);
    if (!p_token) {
	return (T_TOKENNOTFOUND);
    }
    
    /* generate reflection */
    i = t_internal_reflect_token(p_pag, p_token, &outtoken);
    if (i) {
	return (i);
    }
    
    /* add token to pag */
    i = t_internal_add_token_to_pag(outtoken, p_pag);
    if (i) {
	/* if we can't, destroy the reflection */
	p_token->refcount++; /* XXXX HACK to make gc'able */
	t_internal_decref_tokenstr(p_token);
	return(i);
    }
    
    /* return tokenid */
    i = T_COPYOUT(&(outtoken->utoken.tokenid), a.p_tokenid2, tokenid_t);
    return(i);
}

/* delete a token */
static int 
ktoken_deletetoken(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_deletetoken_args a;      
    struct pagstr *p_pag;
    struct ktokenstr *p_token;
    int i;
    
    i = T_COPYIN(uap->args, &a, struct t_deletetoken_args);
    if (i) return(i);
    
    /* find the current PAG */
    t_internal_find_pag_by_proc(&p_pag, p);
    if (!p_pag) {
	return (T_PAGNOTFOUND);
    }
    
    /* find the token to delete */
    t_internal_find_token_by_tokenid(a.tokenid, &p_token);
    if (!p_token) {
	return(T_TOKENNOTFOUND);
    }
    
    /* delete the token */
    i = t_internal_delete_token(TOKEN_CRT_PROCESS, p_pag, p_token);
    return(i);
}

/* find a token matching the specified fields of a sample token */
static int
ktoken_findtoken(p, uap)
    struct proc *p;
    struct t_syscall_args *uap;
{
    struct t_findtoken_args a;
    struct pagstr *p_pag;
    struct ktokenstr *p_token;
    struct utokenstr modeltoken;  /* XXXX stack space? */
    tokenid_t tokenid;
    int i;
    
    i = T_COPYIN(uap->args, &a, struct t_findtoken_args);
    if (i) return(i);
    
    i = T_COPYIN(a.p_token, &modeltoken, struct utokenstr);
    if (i) return(i);

    /* find the current PAG */
    t_internal_find_pag_by_proc(&p_pag, p);
    if (!p_pag) {
	return (T_PAGNOTFOUND);
    }

    KTDEBUG(5, "ktoken.ktoken_findtoken(): pag %lu searching for at least "
	    "%lu, fields %du\n", p_pag->pagid, a.mintokenid, a.whichfields);

    /* perform the search */
    t_internal_search_token(p_pag,
			    a.mintokenid,
			    a.whichfields,
			    &modeltoken,
			    &p_token);
    if (!p_token) {
	/* not found */
	KTDEBUG(5, "ktoken.ktoken_findtoken(): pag %lu search turned "
		"up nothing\n", p_pag->pagid);
	tokenid = 0;
    } else {
	/* found */
	KTDEBUG(5, "ktoken.ktoken_findtoken(): pag %lu search returned token "
		"%lu\n", p_pag->pagid, p_token->utoken.tokenid);

	tokenid = p_token->utoken.tokenid;
    }

    /* return the tokenid */
    i = T_COPYOUT(&(tokenid), a.p_tokenid, tokenid_t);
    return(i);
}

/* enable/disable debugging */
static int
ktoken_debug(p, uap)
struct proc *p;         
struct t_syscall_args *uap;
{       
        struct t_debug_args a;
        int i;

	/* only uid 0 may do this */
	if (p->p_cred->pc_ucred->cr_uid != 0)
		return(EPERM);
        
        i = T_COPYIN(uap->args, &a, struct t_debug_args);
        if (i) return(i);

	/* set level */
	ktdebug = a.debug;

	printf("ktoken.ktoken_debug(): debugging set to %d\n", ktdebug);

	return(0);
}

/* hook fork actions */
void
ktoken_at_fork(struct proc *parent,struct proc *child,int flags)
{
	struct pagstr *p_pag;
	KTDEBUG(5,"ktoken.ktoken_at_fork(): fork occurred from %d to %d "
		"w/flags %d\n", parent->p_pid, child->p_pid, flags);

	/* find the parent's PAG */
	t_internal_find_pag_by_proc(&p_pag, parent);

	if (p_pag) {
	    /* if found */
	    KTDEBUG(1, "ktoken.ktoken_at_fork(): %d has a pag (%lu), "
		    "so am adding child %d to it\n", parent->p_pid, 
		    p_pag->pagid, child->p_pid);

	    /* add child to that pag */
	    t_internal_add_proc_pag(child, p_pag);
	} else {
	    /* if not, nothing */
		KTDEBUG(5, "ktoken.ktoken_at_fork(): %d -> %d fork did not "
			"result in pag inheritence\n", parent->p_pid, 
			child->p_pid);
	}
}

/* hook process exit actions */
void
ktoken_at_exit(struct proc *procp)
{
    struct pagstr *p_pag;
    KTDEBUG(5,"ktoken.ktoken_at_exit(): exit occurred in %d\n",
	    procp->p_pid);

    /* find dying process' PAG */
    t_internal_find_pag_by_proc(&p_pag, procp);

    if (p_pag) {
	/* if found */
	KTDEBUG(1, "ktoken.ktoken_at_exit(): %d has a pag (%lu), so am "
		"removing it\n", procp->p_pid, p_pag->pagid);

	/* remove proc from PAG */
	t_internal_remove_proc_pag(procp, p_pag);
    } else {
	/* if not, nothing */
	KTDEBUG(5, "ktoken.ktoken_at_exit(): %d has no pag, so won't remove "
		"it\n", procp->p_pid);
    }
}

