/*-
 * Copyright (c) 1998 Robert N. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name Robert N. Watson may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: utils.c,v 1.4 1998/07/13 22:35:16 robert Exp $
 */
#include <stdio.h>
#include <errno.h>
#include "types.h"
#include "const.h"
#include "globals.h"
#include "utils.h"

/* utility functions */

/* given a sets name, find the set.  return null if not found
 * badly implemented via a linked list */
struct setstr *getsetbyname(char *name) {
	struct setstr *temp;

#ifdef CAUTIOUS
	if (!name) 
		panic("getsetbyname(): null name\n");
#endif

	for (temp = topset; (temp); temp = temp->next) {
		if (!strcmp(temp->name, name)) {
			return(temp);
		}
	}
	return(0);
}
	
/* given a set and a level name, verify that the set declares the levelname
 * -- assumes organize() has already been called
 */
struct declarestr *getsetlevel(struct setstr *set, char *levelname) {
        struct setbodystr *setbody;

#ifdef CAUTIOUS
	if (!set)
		panic("getsetlevel(): null set\n");
	if (!levelname)
		panic("getsetlevel(): null levelname\n");
#endif

        if (!strcmp(DEFAULT_DECLARE_STRING, levelname)) {
                return &default_declarestr;
        }

        for (setbody = set->declares; (setbody); setbody = setbody->next) {
                if (!strcmp(setbody->declare->name, levelname)) {
                        return(setbody->declare);
                }
        }

        return(0);
}

int getnumsets(void) {
	struct setstr *set;
	int i=0;

	for (set=topset; (set); set=set->next) {
		i++;
	}

	return(i);
}

int getnumdeclares(struct setstr *set) {
	struct setbodystr *setbody;
	int i = 0;

	if (!set)
		return(0);

	for (setbody = set->declares; (setbody); setbody=setbody->next) {
		i++;
	}

	return(i);
}

int incorporate_impl(struct implstr *topimpl) {
	struct implstr *impl;
	struct setstr *set;
	struct declarestr *declare;

	for (impl = topimpl; (impl); impl = impl->next) {
		/* check to see if we have such a set */
#ifdef CAUTIOUS
		if (!impl->setname) panic("incorporate_impl(): null "
			"impl->setname\n");
#endif
		set = getsetbyname(impl->setname);
		if (!set) {
			fprintf(stderr, "Implementation file refers to "
				"unknown set '%s'\n", impl->setname);
			return(0);
		}
		declare = getsetlevel(set, impl->level);
		if (!declare) {
			fprintf(stderr, "Implementation file refers to "
				"unknown level '%s' in set '%s'\n",
				impl->level, impl->setname);
			return(0);
		}
		set->chosen_level = declare;
	}
	return(1); /* ok */
}

int save_impl(char *name) {
	struct setstr *set;
	FILE *out;

#ifdef CAUTIOUS
	if (!name) panic("save_impl(): null name\n");
#endif

	/* attempt to open file */
	out = fopen(name, "w");
	if (!out) {
		return(errno);
	}

	for (set=topset; (set); set=set->next) {
		if (set->chosen_level) {
			fprintf(out, "\"%s\" ", set->name);
			if (set->chosen_level == &default_declarestr)
				fprintf(out, "default;\n");
			else
				fprintf(out, "\"%s\";\n",
					set->chosen_level->name);
		} else
		if (!notdefaultdefault) {
			/* for unedited entries, use default policy */
			fprintf(out, "\"%s\" default;\n", set->name);
		} else {
			/* nothing */
		}
	}

	fclose(out);

	return(0); /* ok */
}

/* apply a particular level to a particular set */
int apply_impl(FILE *applyout, struct setstr *set, char *levelname) {
	struct declarestr *declare;
	struct setbodystr *setbody;
	struct filebodystr *filebody;

	declare = getsetlevel(set, levelname);

#ifdef CAUTIOUS
	/* cautious because one invariant is that the availability of a level
	   to a set has already been verified */
	if (!declare) panic("apply_impl(): null declare\n");
#endif

	for (setbody=set->files; (setbody); setbody=setbody->next) {
		/* iterate list of files */
#ifdef CAUTIOUS
		if (!setbody->filename) panic("apply_impl(): set '%s' has "
			"null filename\n", set->name);
#endif
		/* current file is setbody->filename */
		
		for (filebody=setbody->filebody; (filebody);
			filebody=filebody->next) {
			if (filebody->type != FILE_COMMENT) {
#ifdef CAUTIOUS
				if (!filebody->level) panic("apply_impl(): "
					"set '%s' has "
					"null declare filebody of type %d\n",
					set->name, filebody->type);
#endif
				if (!strcmp(filebody->level, levelname)) {
					switch(filebody->type) {
					case FILE_MODE:
						fprintf(applyout, "chmod %s %s\n",
							filebody->value,
							setbody->filename);
						break;
					case FILE_OWNER:
						fprintf(applyout, "chown %s %s\n",
							filebody->value,
							setbody->filename);
						break;
					case FILE_GROUP:
						fprintf(applyout, "chgrp %s %s\n",
							filebody->value,
							setbody->filename);
						break;
					default:
						panic("apply_impl(): set '%s', file "
							"'%s': invalid type\n",
							set->name, setbody->filename);
					}
				}
			}
		} /* (filebody->type != FILE_COMMENT) */
	}	
	return(1); /* ok */
}

int create_impl_script(char *applyfilename) {
	struct implstr *impl;
	struct setstr *set;
	FILE *applyout;

        applyout = fopen(applyfilename, "w");
        if (!applyout) {
                perror("applysuid");
                exit(-1);
        }

        for (impl = topimpl; (impl); impl=impl->next) {
		set = getsetbyname(impl->setname);
#ifdef CAUTIOUS
		if (!set) panic("create_impl_script(): set '%s' not found\n",
			set->name);
#endif
                if (!apply_impl(applyout, set, impl->level)) {
			fprintf(stderr, "Unable to create implementation for set "
				"'%s'\n", set->name);
			return(0);
		}
	}
	fclose(applyout);
	return(1);
}


