/*-
 * Copyright (c) 1999 Robert N. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *       $Id: $
 */

/*
 * Supporting user-land library code for POSIX.1e auditing
 */

#include <sys/types.h>
#include <sys/queue.h>
#include <sys/audit.h>
#include <sys/time.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

static LIST_HEAD(, audit_private_alloc_str)	audit_private_alloc_head;
struct audit_private_alloc_str {
	LIST_ENTRY(audit_private_alloc_str)	entries;
	void	*objp;
	int	type;
};

static int	audit_private_alloc_ok=0;	/* set to 1 once in use */

static const char *const posix_aud_evlist[] = {
	"none",				/* AUD_AET_NONE */
	"aud_switch",			/* AUD_AET_AUD_SWITCH */
	"aud_write",			/* AUD_AET_AUD_WRITE */
	"chdir",			/* AUD_AET_CHDIR */
	"chmod",			/* AUD_AET_CHMOD */
	"chown",			/* AUD_AET_CHOWN */
	"creat",			/* AUD_AET_CREAT */
	"dup",				/* AUD_AET_DUP */
	"exec",				/* AUD_AET_EXEC */
	"_exit",			/* AUD_AET_EXIT */
	"fork",				/* AUD_AET_FORK */
	"kill",				/* AUD_AET_KILL */
	"link",				/* AUD_AET_LINK */
	"mkdir",			/* AUD_AET_MKDIR */
	"mkfifo",			/* AUD_AET_MKFIFO */
	"open",				/* AUD_AET_OPEN */
	"pipe",				/* AUD_AET_PIPE */
	"rename",			/* AUD_AET_RENAME */
	"setgid",			/* AUD_AET_SETGID */
	"setuid",			/* AUD_AET_SETUID */
	"unlink",			/* AUD_AET_UNLINK */
	"utime",			/* AUD_AET_UTIME */
	"acl_delete_def_file",		/* AUD_AET_ACL_DELETE_DEF_FILE */
	"acl_set_fd",			/* AUD_AET_ACL_SET_FD */
	"acl_set_file",			/* AUD_AET_ACL_SET_FILE */
	"cap_set_fd",			/* AUD_AET_CAP_SET_FD */
	"cap_set_file",			/* AUD_AET_CAP_SET_FILE */
	"cap_set_proc",			/* AUD_AET_CAP_SET_PROC */
	"inf_set_fd",			/* AUD_AET_INF_SET_FD */
	"inf_set_file",			/* AUD_AET_INF_SET_FILE */
	"info_set_proc",		/* AUD_AET_INF_SET_PROC */
	"mac_set_fd",			/* AUD_AET_MAC_SET_FD */
	"mac_set_file",			/* AUD_AET_MAC_SET_FILE */
	"mac_set_proc",			/* AUD_AET_MAC_SET_PROC */
	"bsd_misc",			/* AUD_AET_BSD_MISC */
};
const int posix_aud_nev = sizeof(posix_aud_evlist) /
	sizeof(posix_aud_evlist[0]);

#define MIN(a,b)	(((a)<(b))?(a):(b))


/*
 * Prototypes for syscalls
 */
int aud_switch_sc(int *aud_state);
int aud_write_sc(caddr_t addr, size_t len);
int aud_get_id_sc(pid_t pid, uid_t *uid);

/*
 * A new object has been allocated, and will be exposed to the user.  Register
 * it so we can later recover its type to free it.
 */
static int
audit_private_newobject(int type, void *objp)
{
	struct audit_private_alloc_str	*apas;

	if (audit_private_alloc_ok == 0) {
		audit_private_alloc_ok = 1;
		LIST_INIT(&audit_private_alloc_head);
	}

	apas = (struct audit_private_alloc_str *) malloc(sizeof(
	    struct audit_private_alloc_str));
	if (apas == NULL)
		return(ENOMEM);

	apas->objp = objp;
	apas->type = type;

	LIST_INSERT_HEAD(&audit_private_alloc_head, apas, entries);

	return(0);
}


/*
 * Given an object pointer, return its type, if possible.
 */
static int
audit_private_getobject(int *type, void *objp)
{
	struct audit_private_alloc_str	*apas;

	for (apas = audit_private_alloc_head.lh_first; apas != NULL;
	    apas = apas->entries.le_next) {
		if (apas->objp == objp)
			break;
	}

	if (apas == NULL)
		return(EINVAL);

	if (type != NULL)
		*type = apas->type;
	return(0);
}


/*
 * Given an object pointer, remove it from the list of allocations we manage.
 * Return 0 on success, non-0 on failure.
 */
static int
audit_private_remobject(void *objp)
{
	struct audit_private_alloc_str	*apas;
	
	for (apas = audit_private_alloc_head.lh_first; apas != NULL;
	    apas = apas->entries.le_next) {
		if (apas->objp == objp)
			break;
	}

	if (apas == NULL)
		return(EINVAL);

	LIST_REMOVE(apas, entries);
	free(apas);

	return(0);
}

/*
 * Given an info structure that we allocated, free it and anything associated
 * with it.  For now, we just free first its data pointer, and then it.
 * AUD_TYPE_STRING_ARRAY might demand more serious behavior.
 * XXXXX
 */     
static void     
audit_private_free_info(aud_info_t *info)
{       
 
	free(info->aud_info_p);
	free(info); 
}


/*
 * Recursively free everything associated with an audit record
 * - While headers exist, call aud_delete_hdr() on them
 * - While subjects exist, call aud_delete_subj() on them
 * - While events exist, call aud_delete_event() on them
 * - While objects exist, call aud_delete_obj() on them
 * - free the audrec
 */
static void
audit_private_free_audrec(struct aud_rec_str *rec)
{
	struct aud_evinfo_str	*evinfo;
	struct aud_hdr_str	*hdr;
	struct aud_obj_str	*obj;
	struct aud_subj_str	*subj;
	aud_info_t		*info;

	/* assume type check has been done, or is not needed */

	while (rec->aud_evinfo_head.lh_first != NULL) {
		evinfo = rec->aud_evinfo_head.lh_first;
		LIST_REMOVE(evinfo, aud_evinfos);

		while (evinfo->aud_info_head.lh_first != NULL) {
			info = evinfo->aud_info_head.lh_first;
			LIST_REMOVE(info, aud_infos);

			audit_private_free_info(info);
		}
		free(evinfo);
	}

	while (rec->aud_hdr_head.lh_first != NULL) {
		hdr = rec->aud_hdr_head.lh_first;
		LIST_REMOVE(hdr, aud_hdrs);
	
		while (hdr->aud_info_head.lh_first != NULL) {
			info = hdr->aud_info_head.lh_first;
			LIST_REMOVE(info, aud_infos);
	
			audit_private_free_info(info);
		}

		free(hdr);
	}

	while (rec->aud_obj_head.lh_first != NULL) {
		obj = rec->aud_obj_head.lh_first;
		LIST_REMOVE(obj, aud_objs);

		while (obj->aud_info_head.lh_first != NULL) {
			info = obj->aud_info_head.lh_first;
			LIST_REMOVE(info, aud_infos);

			audit_private_free_info(info);
		}
		free(obj);
	}

	while (rec->aud_subj_head.lh_first != NULL) {
		subj = rec->aud_subj_head.lh_first;
		LIST_REMOVE(subj, aud_subjs);

		while (subj->aud_info_head.lh_first != NULL) {
			info = subj->aud_info_head.lh_first;
			LIST_REMOVE(info, aud_infos);

			audit_private_free_info(info);
		}
		free(subj);
	}

	free(rec);
}


/*
 * Exposed POSIX.1E functions + syscall wrappers
 */

/* 24.4 Functions */

/*
 * Given a buffer (pointed to by aud_rec_ext_p, of size size) and an
 * aud_rec_t, fill the buffer with a flattened version of the record.
 * return the size, or -1 on error.
 */
ssize_t
aud_copy_ext(void *aud_rec_ext_p, aud_rec_t aud_rec_int,
	     ssize_t size)
{
	struct aud_rec_str	*rec;
	struct aud_evinfo_str	*evinfo;
	struct aud_hdr_str	*hdr;
	struct aud_obj_str	*obj;
	struct aud_subj_str	*subj;
	aud_info_t		*info;

	struct flat_header	*f_header;
	struct flat_aud_evinfo	*f_evinfo;
	struct flat_aud_hdr	*f_hdr;
	struct flat_aud_obj	*f_obj;
	struct flat_aud_subj	*f_subj;
	struct flat_aud_info	*f_info;

	ssize_t	finalsize;
	void	*current;

	rec = aud_rec_int;
	if (rec->magic != AUD_MAGIC_AUDREC) {
		errno = EINVAL;
		return((ssize_t)-1);
	}

	finalsize = aud_size(aud_rec_int);
	if (finalsize == -1)
		return((ssize_t)-1);

	if (finalsize > size) {
		errno = ERANGE;
		return((ssize_t)-1);
	}

	current = aud_rec_ext_p;

	/* Flattened Record Header */
	f_header = current;
	f_header->fh_magic = AUD_MAGIC_FLAT;
	f_header->fh_length = finalsize - sizeof(struct flat_header);
	f_header->fh_version = AUD_FLAT_VERSION;
	f_header->fh_uid = -1;	/* just a dummy value */
	current += sizeof(struct flat_header);

	/*
	 * Now write out each set, in the following order (alphabetic):
	 *     evinfo
	 *     hdr
	 *     obj
	 *     subj
	 */

	for (evinfo = rec->aud_evinfo_head.lh_first; evinfo != NULL;
	    evinfo = evinfo->aud_evinfos.le_next) {
		f_evinfo = current;
		f_evinfo->fae_magic = AUD_MAGIC_FEVINFO;
		f_evinfo->fae_num_info = evinfo->aud_info_num;

		current += sizeof(struct flat_aud_evinfo);

		for (info = evinfo->aud_info_head.lh_first; info != NULL;
		    info = info->aud_infos.le_next) {
			f_info = current;

			f_info->fai_magic = AUD_MAGIC_FINFO;
			f_info->fai_type = info->aud_info_type;
			f_info->fai_item_id = info->item_id;
			f_info->fai_length = info->aud_info_length;
			current += sizeof(struct flat_aud_info);

			/* copy in the info */
			memcpy(current, info->aud_info_p,
			    info->aud_info_length);
			current += info->aud_info_length;
		}
	}
	
	for (hdr = rec->aud_hdr_head.lh_first; hdr != NULL;
	    hdr = hdr->aud_hdrs.le_next) {
		f_hdr = current;
		f_hdr->fah_magic = AUD_MAGIC_FHDR;
		f_hdr->fah_num_info = hdr->aud_info_num;

		current += sizeof(struct flat_aud_hdr);
	
		for (info = hdr->aud_info_head.lh_first; info != NULL;
		    info = info->aud_infos.le_next) {
			f_info = current;

			f_info->fai_magic = AUD_MAGIC_FINFO;
			f_info->fai_type = info->aud_info_type;
			f_info->fai_item_id = info->item_id;
			f_info->fai_length = info->aud_info_length;
			current += sizeof(struct flat_aud_info);

			/* copy in the info */
			memcpy(current, info->aud_info_p,
			    info->aud_info_length);
			current += info->aud_info_length;
		}
	}

	for (obj = rec->aud_obj_head.lh_first; obj != NULL;
	    obj = obj->aud_objs.le_next) {
		f_obj = current;
		f_obj->fao_magic = AUD_MAGIC_FOBJ;
		f_obj->fao_num_info = obj->aud_info_num;

		current += sizeof(struct flat_aud_obj);

		for (info = obj->aud_info_head.lh_first; info != NULL;
		    info = info->aud_infos.le_next) {
			f_info = current;

			f_info->fai_magic = AUD_MAGIC_FINFO;
			f_info->fai_type = info->aud_info_type;
			f_info->fai_item_id = info->item_id;
			f_info->fai_length = info->aud_info_length;
			current += sizeof(struct flat_aud_info);

			/* copy in the info */
			memcpy(current, info->aud_info_p,
			    info->aud_info_length);
			current += info->aud_info_length;
		}
	}

	for (subj = rec->aud_subj_head.lh_first; subj != NULL;
	    subj = subj->aud_subjs.le_next) {
		f_subj = current;
		f_subj->fas_magic = AUD_MAGIC_FSUBJ;
		f_subj->fas_num_info = subj->aud_info_num;

		current += sizeof(struct flat_aud_subj);

		for (info = subj->aud_info_head.lh_first; info != NULL;
		    info = info->aud_infos.le_next) {
			f_info = current;

			f_info->fai_magic = AUD_MAGIC_FINFO;
			f_info->fai_type = info->aud_info_type;
			f_info->fai_item_id = info->item_id;
			f_info->fai_length = info->aud_info_length;
			current += sizeof(struct flat_aud_info);

			/* copy in the info */
			memcpy(current, info->aud_info_p,
			    info->aud_info_length);
			current += info->aud_info_length;
		}
	}

	if (current - aud_rec_ext_p != finalsize) {
		/* This can only happen as the result of memory corruption! */
		errno = ERANGE;
		return(-1);
	}

	return(finalsize);
}

/*
 * Used internally to prevent nasties; the exposed call, per POSIX.1E, does
 * not allow the passing of a bound on the flattened record size.  This means
 * that a corrupted record could result in tromped memory or a segfault.
 * For the purposes of aud_read, this is used.  If a bound of size 0 is seen,
 * the bound is ignored.
 */
aud_rec_t
audit_copy_int_np(const void *aud_rec_ext_p, ssize_t size)
{
	struct aud_rec_str	*rec;
	aud_evinfo_t	evinfo;
	aud_hdr_t	hdr;
	aud_obj_t	obj;
	aud_subj_t	subj;
	aud_info_t	info;

	const struct flat_header	*f_header;
	const struct flat_aud_evinfo	*f_evinfo;
	const struct flat_aud_hdr	*f_hdr;
	const struct flat_aud_obj	*f_obj;
	const struct flat_aud_subj	*f_subj;
	const struct flat_aud_info	*f_info;
	const void	*current, *end;

	int	i;
	const int	*magic;

	if ((size != 0) && (size < sizeof(struct flat_header))) {
		errno = EINVAL;
		return(NULL);
	}

	/* Flattened Record Header */

	current = aud_rec_ext_p;
	f_header = current;

	if ((f_header->fh_magic != AUD_MAGIC_FLAT) ||
	    (f_header->fh_version != AUD_FLAT_VERSION)) {
		errno = EINVAL;
		return(NULL);
	}

	if ((size != 0)
	    && (f_header->fh_length != size - sizeof(struct flat_header))) {
		errno = EINVAL;
		return(NULL);
	}

	end = current + sizeof(struct flat_header) + f_header->fh_length;
	current += sizeof(struct flat_header);

	/* Allocate an aud_rec_str */

	rec = (struct aud_rec_str *) aud_init_record();
	if (rec == NULL) { 
		errno = ENOMEM;
		return(NULL);
	}

	/* Now a series of 
	 *     evinfo
	 *     hdr
	 *     obj
	 *     subj
	 */

#define RETURN(x, error) {				\
				aud_free(rec);		\
				if (error != 0)		\
					errno = error;	\
				return(x);		\
			}

	while (current + sizeof(int) < end) {
		magic = current;
		switch(*magic) {
		case AUD_MAGIC_FEVINFO:
			f_evinfo = current;
			current += sizeof(struct flat_aud_evinfo);
			if (current > end)
				RETURN(NULL, EINVAL);

			i = aud_put_event(rec, NULL, &evinfo);
			if (i == -1)
				RETURN(NULL, 0);

			magic = current;
			while ((current + sizeof(int) < end) &&
			    (*magic == AUD_MAGIC_FINFO)) {
				f_info = current;
				current += sizeof(struct flat_aud_info);
				if (current > end)
					RETURN(NULL, EINVAL);

				info.aud_info_type = f_info->fai_type;
				info.aud_info_length = f_info->fai_length;
				info.aud_info_p = current;

				current += f_info->fai_length;
				if (current > end)
					RETURN(NULL, EINVAL);

				i = aud_put_event_info(evinfo, AUD_LAST_ITEM,
				    f_info->fai_item_id, &info);
				if (i == -1)
					RETURN(NULL, 0);
				magic = current;
			}

			break;

		case AUD_MAGIC_FHDR:
			f_hdr = current;
			current += sizeof(struct flat_aud_hdr);
			if (current > end)
				RETURN(NULL, EINVAL);

			i = aud_put_hdr(rec, NULL, &hdr);
			if (i == -1)
				RETURN(NULL, 0);

			magic = current;
			while ((current + sizeof(int) < end) &&
			    (*magic == AUD_MAGIC_FINFO)) {
				f_info = current;
				current += sizeof(struct flat_aud_info);
				if (current > end)
					RETURN(NULL, EINVAL);

				info.aud_info_type = f_info->fai_type;
				info.aud_info_length = f_info->fai_length;
				info.aud_info_p = current;

				current += f_info->fai_length;
				if (current > end)
					RETURN(NULL, EINVAL);

				i = aud_put_hdr_info(hdr, AUD_LAST_ITEM,
				    f_info->fai_item_id, &info);
				if (i == -1)
					RETURN(NULL, 0);
				magic = current;
			}
			break;

		case AUD_MAGIC_FOBJ:
			f_obj = current;
			current += sizeof(struct flat_aud_obj);
			if (current > end)
				RETURN(NULL, EINVAL);

			i = aud_put_obj(rec, NULL, &obj);
			if (i == -1)
				RETURN(NULL, 0);

			magic = current;
			while ((current + sizeof(int) < end) &&
			    (*magic == AUD_MAGIC_FINFO)) {
				f_info = current;
				current += sizeof(struct flat_aud_info);
				if (current > end)
					RETURN(NULL, EINVAL);

				info.aud_info_type = f_info->fai_type;
				info.aud_info_length = f_info->fai_length;
				info.aud_info_p = current;

				current += f_info->fai_length;
				if (current > end)
					RETURN(NULL, EINVAL);

				i = aud_put_obj_info(obj, AUD_LAST_ITEM,
				    f_info->fai_item_id, &info);
				if (i == -1)
					RETURN(NULL, 0);
				magic = current;
			}
			break;

		case AUD_MAGIC_FSUBJ:
			f_subj = current;
			current += sizeof(struct flat_aud_subj);
			if (current > end)
				RETURN(NULL, EINVAL);

			i = aud_put_subj(rec, NULL, &subj);
			if (i == -1)
				RETURN(NULL, 0);

			magic = current;
			while ((current + sizeof(int) < end) &&
			    (*magic == AUD_MAGIC_FINFO)) {
				f_info = current;
				current += sizeof(struct flat_aud_info);
				if (current > end)
					RETURN(NULL, EINVAL);

				info.aud_info_type = f_info->fai_type;
				info.aud_info_length = f_info->fai_length;
				info.aud_info_p = current;

				current += f_info->fai_length;
				if (current > end)
					RETURN(NULL, EINVAL);

				i = aud_put_subj_info(subj, AUD_LAST_ITEM,
				    f_info->fai_item_id, &info);
				if (i == -1)
					RETURN(NULL, 0);
				magic = current;
			}
			break;

		case AUD_MAGIC_FINFO:
		default:
			/* this is not expected */
			RETURN(NULL, EINVAL);
		}
	}
#undef RETURN

	return((aud_rec_t)rec);
}

/*
 * Turn a data blob back into an audit record -- wrapped to ignore size
 * of the blob.
 */
aud_rec_t
aud_copy_int(const void *aud_rec_ext_p)
{

	return(audit_copy_int_np(aud_rec_ext_p, 0));
}


int
aud_delete_event(aud_evinfo_t aud_event_d)
{
	struct aud_evinfo_str	*evinfo;
	aud_info_t	*info;

	evinfo = (struct aud_evinfo_str *) aud_event_d;
	if (evinfo->magic != AUD_MAGIC_EVINFO) {
		errno = EINVAL;
		return(-1);
	}

	/* Remove references from any aud_rec_t's */
	
	LIST_REMOVE(evinfo, aud_evinfos);

	/*
	 * Free all children info structures
	 */
	while (evinfo->aud_info_head.lh_first != NULL) {
		info = evinfo->aud_info_head.lh_first;
		LIST_REMOVE(info, aud_infos);
		audit_private_free_info(info);
	}

	evinfo->magic = 0;
	free(evinfo);

	return(0);
}


int
aud_delete_event_info(aud_evinfo_t aud_event_d, int item_id)
{
	struct aud_evinfo_str	*evinfo;
	aud_info_t	*info;

	evinfo = (struct aud_evinfo_str *) aud_event_d;
	if (evinfo->magic != AUD_MAGIC_EVINFO) {
		errno = EINVAL;
		return(-1);
	}

	for (info = evinfo->aud_info_head.lh_first;
	    info != NULL; info = info->aud_infos.le_next) {
		if (info->item_id == item_id) {
			if (evinfo->aud_info_cur == info)
				evinfo->aud_info_cur = NULL;
			LIST_REMOVE(info, aud_infos);
			audit_private_free_info(info);
			evinfo->aud_info_num--;
			return(0);
		}
	}

	/* not found */
	errno = EINVAL;
	return(-1);
}


int
aud_delete_hdr(aud_hdr_t aud_hdr_d)
{
	struct aud_hdr_str	*hdr;
	aud_info_t	*info;

	hdr = (struct aud_hdr_str *) aud_hdr_d;
	if (hdr->magic != AUD_MAGIC_HDR) {
		errno = EINVAL;
		return(-1);
	}

	/* Remove references from any aud_rec_t's */

	LIST_REMOVE(hdr, aud_hdrs);

	/*
	 * Free all children info structures
	 */
	while (hdr->aud_info_head.lh_first != NULL) {
		info = hdr->aud_info_head.lh_first;
		LIST_REMOVE(info, aud_infos);
		audit_private_free_info(info);
	}

	hdr->magic = 0;
	free(hdr);

	return(0);
}


int
aud_delete_hdr_info(aud_hdr_t aud_hdr_d, int item_id)
{
	struct aud_hdr_str	*hdr;
	aud_info_t	*info;

	hdr = (struct aud_hdr_str *) aud_hdr_d;
	if (hdr->magic != AUD_MAGIC_HDR) {
		errno = EINVAL;
		return(-1);
	}

	for (info = hdr->aud_info_head.lh_first;
	    info != NULL; info = info->aud_infos.le_next) {
		if (info->item_id == item_id) {
			if (hdr->aud_info_cur == info)
				hdr->aud_info_cur = NULL;
			LIST_REMOVE(info, aud_infos);
			audit_private_free_info(info);
			hdr->aud_info_num--;
			return(0);
		}
	}

	/* not found */
	errno = EINVAL;
	return(-1);
}


int
aud_delete_obj(aud_obj_t aud_obj_d)
{
	struct aud_obj_str	*obj;
	aud_info_t	*info;

	obj = (struct aud_obj_str *) aud_obj_d;
	if (obj->magic != AUD_MAGIC_OBJ) {
		errno = EINVAL;
		return(-1);
	}

	/* Remove references from any aud_rec_t's */

	LIST_REMOVE(obj, aud_objs);

	/*
	 * Free all children info structures
	 */
	while (obj->aud_info_head.lh_first != NULL) {
		info = obj->aud_info_head.lh_first;
		LIST_REMOVE(info, aud_infos);
		audit_private_free_info(info);
	}

	obj->magic = 0;
	free(obj);

	return(0);
}


int
aud_delete_obj_info(aud_obj_t aud_obj_d, int item_id)
{
	struct aud_obj_str	*obj;
	aud_info_t	*info;

	obj = (struct aud_obj_str *) aud_obj_d;
	if (obj->magic != AUD_MAGIC_OBJ) {
		errno = EINVAL;
		return(-1);
	}
		

	for (info = obj->aud_info_head.lh_first;
	    info != NULL; info = info->aud_infos.le_next) {
		if (info->item_id == item_id) {
			if (obj->aud_info_cur == info)
				obj->aud_info_cur = NULL;
			LIST_REMOVE(info, aud_infos);
			audit_private_free_info(info);
			obj->aud_info_num--;
			return(0);
		}
	}

	/* not found */
	errno = EINVAL;
	return(-1);
}


int
aud_delete_subj(aud_subj_t aud_subj_d)
{
	struct aud_subj_str	*subj;
	aud_info_t	*info;

	subj = (struct aud_subj_str *) aud_subj_d;
	if (subj->magic != AUD_MAGIC_SUBJ) {
		errno = EINVAL;
		return(-1);
	}

	/* Remove references from any aud_rec_t's */

	LIST_REMOVE(subj, aud_subjs);

	/*
	 * Free all children info structures
	 */
	while(subj->aud_info_head.lh_first != NULL) {
		info = subj->aud_info_head.lh_first;
		LIST_REMOVE(info, aud_infos);
		audit_private_free_info(info);
	}

	subj->magic = 0;
	free(subj);
	
	return(0);
}


int
aud_delete_subj_info(aud_subj_t aud_subj_d, int item_id)
{
	struct aud_subj_str	*subj;
	aud_info_t	*info;

	subj = (struct aud_subj_str *) aud_subj_d;
	if (subj->magic != AUD_MAGIC_SUBJ) {
		errno = EINVAL;
		return(-1);
	}

	for (info = subj->aud_info_head.lh_first;
	    info != NULL; info = info->aud_infos.le_next) {
		if (info->item_id == item_id) {
			if (subj->aud_info_cur == info)
				subj->aud_info_cur = NULL;
			LIST_REMOVE(info, aud_infos);
			audit_private_free_info(info);
			subj->aud_info_num--;
			return(0);
		}
	}
			
	/* not found */
	errno = EINVAL;
	return(-1);
}


/*
 * Duplicate an audit record
 * XXXX -- relies on get_*_info to copy out item_id into the info_t that is
 * returned.  This is a non-standard field, and non-standard behavior, but
 * it makes things a lot smoother.  It should only be used in the library,
 * and not elsewhere.
 */
aud_rec_t
aud_dup_record(aud_rec_t ar)
{
	struct aud_rec_str	*rec, *newrec;
	aud_evinfo_t		evinfo_new, evinfo_old;
	aud_hdr_t		hdr_new, hdr_old;
	aud_obj_t		obj_new, obj_old;
	aud_subj_t		subj_new, subj_old;
	aud_info_t		info;

	int	i, count, infocount, num_evinfo, num_hdr, num_obj, num_subj,
		num_info;

	rec = ar;
	if (rec->magic != AUD_MAGIC_AUDREC) {
		errno = EINVAL;
		return((aud_rec_t)NULL);
	}

	num_evinfo = aud_get_event(rec, 0, NULL);
	if (num_evinfo == -1)
		return(NULL);
	num_hdr = aud_get_hdr(rec, 0, NULL);
	if (num_hdr == -1)
		return(NULL);
	num_obj = aud_get_obj(rec, 0, NULL);
	if (num_obj == -1)
		return(NULL);
	num_subj = aud_get_subj(rec, 0, NULL);
	if (num_subj == -1)
		return(NULL);

	newrec = aud_init_record();
	if (!newrec) {
		/* errno already set */
		return((aud_rec_t)NULL);
	}

#define RETURN(ptr, error) {			\
			aud_free(newrec);	\
			if (error != 0)		\
				errno = error;	\
			return(ptr);		\
			}

	for (count = 1; count <= num_evinfo; count++) {
		i = aud_get_event(rec, count, &evinfo_old);
		if (i == -1)
			RETURN(NULL, 0);
	
		i = aud_put_event(newrec, NULL, &evinfo_new);
		if (i == -1)
			RETURN(NULL, 0);

		num_info = aud_get_event_info(evinfo_old, 0, NULL);
		if (num_info == -1)
			RETURN(NULL, 0);

		if (num_info) {
			i = aud_get_event_info(evinfo_old, AUD_FIRST_ITEM,
			    &info);
			if (i == -1)
				RETURN(NULL, 0);
			/*
			 * XXXX Rely in info->item_id being setting; this is
			 * not to spec, but the spec provides no way of
			 * retrieving the item_id of an info retrieved using
			 * _FIRST and _NEXT.
			 */
			i = aud_put_event_info(evinfo_new, AUD_LAST_ITEM,
			    info.item_id, &info);
			if (i == -1)
				RETURN(NULL, 0);

			for (infocount = 2; infocount <= num_info;
			    infocount++) {
				i = aud_get_event_info(evinfo_old,
				    AUD_NEXT_ITEM, &info);
				if (i == -1)
					RETURN(NULL, 0);
				i = aud_put_event_info(evinfo_new,
				    AUD_LAST_ITEM, info.item_id, &info);
				if (i == -1)
					RETURN(NULL, 0);
			}
		}
	}

	for (count = 1; count <= num_hdr; count++) {
		i = aud_get_hdr(rec, count, &hdr_old);
		if (i == -1)
			RETURN(NULL, 0);

		i = aud_put_hdr(newrec, NULL, &hdr_new);
		if (i == -1)
			RETURN(NULL, 0);

		num_info = aud_get_hdr_info(hdr_old, 0, NULL);
		if (num_info == -1)
			RETURN(NULL, 0);

		if (num_info) {
			i = aud_get_hdr_info(hdr_old, AUD_FIRST_ITEM,
			    &info);
			if (i == -1)
				RETURN(NULL, 0);
			/*
			 * XXXX Rely in info->item_id being setting; this is
			 * not to spec, but the spec provides no way of
			 * retrieving the item_id of an info retrieved using
			 * _FIRST and _NEXT.
			 */
			i = aud_put_hdr_info(hdr_new, AUD_LAST_ITEM,
			    info.item_id, &info);
			if (i == -1)
				RETURN(NULL, 0);

			for (infocount = 2; infocount <= num_info;
			    infocount++) {
				i = aud_get_hdr_info(hdr_old,
				    AUD_NEXT_ITEM, &info);
				if (i == -1)
					RETURN(NULL, 0);
				i = aud_put_hdr_info(hdr_new,
				    AUD_LAST_ITEM, info.item_id, &info);
				if (i == -1)
					RETURN(NULL, 0);
			}
		}
	}

	for (count = 1; count <= num_obj; count++) {
		i = aud_get_obj(rec, count, &hdr_old);
		if (i == -1)
			RETURN(NULL, 0);
		i = aud_put_obj(newrec, NULL, &obj_new);
		if (i == -1)
			RETURN(NULL, 0);

		num_info = aud_get_obj_info(obj_old, 0, NULL);
		if (num_info == -1)
			RETURN(NULL, 0);

		if (num_info) {
			i = aud_get_obj_info(obj_old, AUD_FIRST_ITEM,
			    &info);
			if (i == -1)
				RETURN(NULL, 0);
			/*
			 * XXXX Rely in info->item_id being setting; this is
			 * not to spec, but the spec provides no way of
			 * retrieving the item_id of an info retrieved using
			 * _FIRST and _NEXT.
			 */
			i = aud_put_obj_info(obj_new, AUD_LAST_ITEM,
			    info.item_id, &info);
			if (i == -1)
				RETURN(NULL, 0);

			for (infocount = 2; infocount <= num_info;
			    infocount++) {
				i = aud_get_obj_info(obj_old,
				    AUD_NEXT_ITEM, &info);
				if (i == -1)
					RETURN(NULL, 0);
				i = aud_put_obj_info(obj_new,
				    AUD_LAST_ITEM, info.item_id, &info);
				if (i == -1)
					RETURN(NULL, 0);
			}
		}
	}

	for (count = 1; count <= num_subj; count++) {
		i = aud_get_subj(rec, count, &hdr_old);
		if (i == -1)
			RETURN(NULL, 0);
		i = aud_put_subj(newrec, NULL, &evinfo_new);
		if (i == -1)
			RETURN(NULL, 0);

		num_info = aud_get_subj_info(subj_old, 0, NULL);
		if (num_info == -1)
			RETURN(NULL, 0);

		if (num_info) {
			i = aud_get_subj_info(subj_old, AUD_FIRST_ITEM,
			    &info);
			if (i == -1)
				RETURN(NULL, 0);
			/*
			 * XXXX Rely in info->item_id being setting; this is
			 * not to spec, but the spec provides no way of
			 * retrieving the item_id of an info retrieved using
			 * _FIRST and _NEXT.
			 */
			i = aud_put_subj_info(subj_new, AUD_LAST_ITEM,
			    info.item_id, &info);
			if (i == -1)
				RETURN(NULL, 0);

			for (infocount = 2; infocount <= num_info;
			    infocount++) {
				i = aud_get_subj_info(subj_old,
				    AUD_NEXT_ITEM, &info);
				if (i == -1)
					RETURN(NULL, 0);
				i = aud_put_subj_info(subj_new,
				    AUD_LAST_ITEM, info.item_id, &info);
				if (i == -1)
					RETURN(NULL, 0);
			}
		}
	}

#undef RETURN

	return(newrec);
}


/*
 * Given a textual representation of an event type, return the type number
 */
int
aud_evid_from_text(const char *text)
{
	int	i;

	for (i=AUD_AET_FIRST; i<=AUD_AET_LAST; i++)
		if (!strcmp(text, posix_aud_evlist[i]))
			return(i);
			
	errno = EINVAL;
	return(-1);
}


/*
 * Given an integer event_type, return a string for it
 */
char
*aud_evid_to_text(int event_type, ssize_t *aud_info_length)
{
	char	*ch;
	int	err;

	if ((event_type < AUD_AET_FIRST) || (event_type > AUD_AET_LAST)) {
		errno = EINVAL;
		return((char *)NULL);
	}

	ch = strdup(posix_aud_evlist[event_type]);
	if (!ch) {
		errno = ENOMEM;
		return((char *)NULL);
	}

	err = audit_private_newobject(AUD_TYPE_STRING, ch);
	if (err) {
		free(ch);
		errno = err;
		return((char *)NULL);
	}
	
	*aud_info_length = strlen(ch);
	return(ch);
}


/*
 * free an allocated object: check with type manager to see if it is
 * registered.  If so, call a recursive free on it using the type, and
 * unregister.
 */
int
aud_free(void *obj_p)
{
	int	i, type;

	i = audit_private_getobject(&type, obj_p);
	if (i) {
		errno = EINVAL;
		return(-1);
	}

	/* unregister */
	i = audit_private_remobject(obj_p);
	if (i) {
		/* XXXX -- was unable to free reference, but had one? */
	}

	switch(type) {
	case AUD_TYPE_IMP_AUD_REC: 

		audit_private_free_audrec((struct aud_rec_str *)obj_p);
		break;

	case AUD_TYPE_IMP_EVID:
		/* nothing, is a const */
		break;

	case AUD_TYPE_STRING:
		free(obj_p);
		break;

	default:
		/* XXXX */
		errno = EINVAL;
		return(-1);
	}

	return(0);
}


/*
 * Return a list of AET types.  This assumes the list begins at AUD_AET_FIRST
 * and goes through AUD_AET_LAST, and that these are contiguous.  Doh.
 * XXXXX
 */
int
*aud_get_all_evid(void)
{
	int	*buf;
	int	i, num;

	num = AUD_AET_LAST - AUD_AET_FIRST;

	buf = (int *) malloc(sizeof(int) * num + 1);
	if (!buf) {
		errno = ENOMEM;
		return(NULL);
	}

	i = audit_private_newobject(AUD_TYPE_IMP_EVID, buf);
	if (i) {
		free(buf);
		return(NULL);
	}

	for (i = AUD_AET_FIRST; i <= AUD_AET_LAST; i++) {
		buf[i - AUD_AET_FIRST] = i;
	}
	buf[num+1 - AUD_AET_LAST] = -1;

	return(buf);
}


/*
 * given an ar and an index, determine if an evinfo exists matching the
 * index in the ar.  If so, return a new or existing descriptor depending
 * on whether one already exists.  (See libposix1e notes for whether this
 * is accurate according to the draft)
 */
int
aud_get_event(aud_rec_t ar, int index, aud_evinfo_t *aud_event_p)
{
	struct aud_evinfo_str	*evinfo;
	struct aud_rec_str	*rec;
	int	count;

	rec = ar;
	if (rec->magic != AUD_MAGIC_AUDREC) {
		errno = EINVAL;
		return(-1);
	}

	/* if aud_event_p is NULL, just return a count */
	if (aud_event_p == NULL)
		return(rec->aud_num_evinfo);

	/* locate the appropriate index */
	count = 0;
	for (evinfo = rec->aud_evinfo_head.lh_first;
	    evinfo != NULL; evinfo = evinfo->aud_evinfos.le_next) {
		count++;
		if (count == index) {
			/* got a descriptor */
			*aud_event_p = evinfo;
			return(rec->aud_num_evinfo);
		}
	}

	/* couldn't find it, invalid index */
	errno = EINVAL;
	return(-1);
}


/*
 * Return an info field from an audit event, given an item_id.
 */
int
aud_get_event_info(aud_evinfo_t aud_event_d, int item_id,
		   aud_info_t *aud_event_info_p)
{
	struct aud_evinfo_str	*evinfo;
	aud_info_t	*info;

	evinfo = (struct aud_evinfo_str *) aud_event_d;
	if (evinfo->magic != AUD_MAGIC_EVINFO) {
		errno = EINVAL;
		return(-1);
	}

	if (aud_event_info_p == NULL)
		return(evinfo->aud_info_num);

	switch(item_id) {
	case AUD_FIRST_ITEM:
		info = evinfo->aud_info_head.lh_first;
		break;

	case AUD_NEXT_ITEM:
		if (evinfo->aud_info_cur == NULL) {
			errno = EINVAL;
			return(-1);
		}
		info = evinfo->aud_info_cur->aud_infos.le_next;
		break;

	default:
		/* return the matching item_id info record */
		for (info = evinfo->aud_info_head.lh_first;
		    info != NULL; info = info->aud_infos.le_next) {
			if (info->item_id == item_id)
				break;
		}
	}

	if (!info) {
		errno = EINVAL;
		return(-1);
	}

	evinfo->aud_info_cur = info;
	aud_event_info_p->aud_info_type = info->aud_info_type;
	aud_event_info_p->aud_info_length = info->aud_info_length;
	aud_event_info_p->aud_info_p = info->aud_info_p;
	/* XXXX non-standard, but useful for aud_dup_record */
	aud_event_info_p->item_id = info->item_id;
	return(evinfo->aud_info_num);
}


/*
 * Given an ar and an index, determine if a hdr exists matching the index.
 * If so, return a {new,existing} descriptor for it.
 */
int
aud_get_hdr(aud_rec_t ar, int index, aud_hdr_t *aud_hdr_p)
{
	struct aud_hdr_str	*hdr;
	struct aud_rec_str	*rec;
	int	count;

	rec = ar;
	if (rec->magic != AUD_MAGIC_AUDREC) {
		errno = EINVAL;
		return(-1);
	}

	/* if aud_hdr_p is NULL, just return a count */
	if (aud_hdr_p == NULL)
		return(rec->aud_num_hdr);


	/* locate the appropriate index */
	count = 0;
	for (hdr = rec->aud_hdr_head.lh_first;
	    hdr != NULL; hdr = hdr->aud_hdrs.le_next) {
		count++;
		if (count == index) {
			*aud_hdr_p = hdr;
			return(rec->aud_num_hdr);
		}
	}

	/* couldn't find it, invalid index */
	errno = EINVAL;
	return(-1);
}


/* 
 * Return an info field from an audit header, given an item_id
 */
int
aud_get_hdr_info(aud_hdr_t aud_hdr_d, int item_id,
		 aud_info_t *aud_hdr_info_p)
{
	struct aud_hdr_str	*hdr;
	aud_info_t	*info;

	hdr = (struct aud_hdr_str *) aud_hdr_d;
	if (hdr->magic != AUD_MAGIC_HDR) {
		errno = EINVAL;
		return(-1);
	}

	if (aud_hdr_info_p == NULL)
		return(hdr->aud_info_num);

	switch(item_id) {
	case AUD_FIRST_ITEM:
		info = hdr->aud_info_head.lh_first;
		break;

	case AUD_NEXT_ITEM:
		if (hdr->aud_info_cur == NULL) {
			errno = EINVAL;
			return(-1);
		}

		info = hdr->aud_info_cur->aud_infos.le_next;
		break;

	default:
		/* return the matching item_id info record */
		for (info = hdr->aud_info_head.lh_first;
		    info != NULL; info = info->aud_infos.le_next) {
			if (info->item_id == item_id)
				break;
		}
	}

	if (!info) {
		errno = EINVAL;
		return(-1);
	}

	hdr->aud_info_cur = info;
	aud_hdr_info_p->aud_info_type = info->aud_info_type;
	aud_hdr_info_p->aud_info_length = info->aud_info_length;
	aud_hdr_info_p->aud_info_p = info->aud_info_p;
	/* XXXX non-standard, but useful for aud_dup_record */
	aud_hdr_info_p->item_id = info->item_id;    
	return(hdr->aud_info_num);
}


/*
 * Return the aud_id_t for a given process
 * Rely on the syscall to do magic
 * XXXX This could probably use procfs
 */
aud_id_t
aud_get_id(pid_t pid)
{
	uid_t	uid;
	int	i;

	i = aud_get_id_sc(pid, &uid);
	if (i != 0) {
		return((aud_id_t)-1);
	} else
		return(uid);
}


int
aud_get_obj(aud_rec_t ar, int index, aud_obj_t *aud_obj_p)
{
	struct aud_obj_str	*obj;
	struct aud_rec_str	*rec;
	int	count;

	rec = ar;
	if (rec->magic != AUD_MAGIC_AUDREC) {
		errno = EINVAL;
		return(-1);
	}

	/* if aud_obj_p is NULL, just return a count */
	if (aud_obj_p == NULL)
		return(rec->aud_num_obj);   

	/* locate the appropriate index */
	count = 0;
	for (obj = rec->aud_obj_head.lh_first;
	    obj != NULL; obj = obj->aud_objs.le_next) {
		count++;
		if (count == index) {
			*aud_obj_p = obj;
			return(rec->aud_num_obj);
		}
	}

	/* couldn't find it, invalid index */
	errno = EINVAL;
	return(-1);
}


/*
 * Return an info field from an audit event, given an item_d.
 */
int
aud_get_obj_info(aud_obj_t aud_obj_d, int item_id,
		 aud_info_t *aud_obj_info_p)
{
	struct aud_obj_str	*obj;
	aud_info_t	*info;

	obj = (struct aud_obj_str *) aud_obj_d;
	if (obj->magic != AUD_MAGIC_OBJ) {
		errno = EINVAL;
		return(-1);
	}

	if (aud_obj_info_p == NULL)
		return(obj->aud_info_num);

	switch(item_id) {
	case AUD_FIRST_ITEM:
		info = obj->aud_info_head.lh_first;
		break;

	case AUD_NEXT_ITEM:
		if (obj->aud_info_cur == NULL) {
			errno = EINVAL;
			return(-1);
		}
		info = obj->aud_info_cur->aud_infos.le_next;
		break;

	default:
		/* return the matching item_id info record */
		for (info = obj->aud_info_head.lh_first;
		    info != NULL; info = info->aud_infos.le_next) {
			if (info->item_id == item_id) 
				break;
		}
	}

	if (!info) {
		errno = EINVAL;
		return(-1);
        }

        obj->aud_info_cur = info;
        aud_obj_info_p->aud_info_type = info->aud_info_type;
        aud_obj_info_p->aud_info_length = info->aud_info_length;
        aud_obj_info_p->aud_info_p = info->aud_info_p;
        /* XXXX non-standard, but useful for aud_dup_record */
        aud_obj_info_p->item_id = info->item_id;    
        return(obj->aud_info_num);
}


/*
 * given an ar and an index, determine if a subj exists matching the index.
 * If so, return a {new,existing} descriptor for it.
 */
int
aud_get_subj(aud_rec_t ar, int index, aud_subj_t *aud_subj_p)
{
	struct aud_subj_str	*subj;
	struct aud_rec_str	*rec;
	int	count;

	rec = ar;
	if (rec->magic != AUD_MAGIC_AUDREC) {
		errno = EINVAL;
		return(-1);
	}

	/* if aud_subj_p is NULL, just return a count */
	if (aud_subj_p == NULL)
		return(rec->aud_num_subj);   


	/* locate the appropriate index */
	count = 0;
	for (subj = rec->aud_subj_head.lh_first;
	    subj != NULL; subj = subj->aud_subjs.le_next) {
		count++;
		if (count == index) {
			*aud_subj_p = subj;
			return(rec->aud_num_subj);
		}
	}

	/* couldn't find it, invalid index */
	errno = EINVAL;
	return(-1);
}


/*
 * Return an info field from an audit header, given an item_id
 */
int
aud_get_subj_info(aud_subj_t aud_subj_d, int item_id,
		  aud_info_t *aud_subj_info_p)
{
	struct aud_subj_str	*subj;
	aud_info_t	*info;

	subj = (struct aud_subj_str *) aud_subj_d;
	if (subj->magic != AUD_MAGIC_SUBJ) {
		errno = EINVAL;
		return(-1);
	}

	if (aud_subj_info_p == NULL)
		return(subj->aud_info_num);

	switch(item_id) {
	case AUD_FIRST_ITEM:
		info = subj->aud_info_head.lh_first;
		break;

	case AUD_NEXT_ITEM:
		if (subj->aud_info_cur == NULL) {
			errno = EINVAL;
			return(-1);
		}
		info = subj->aud_info_cur->aud_infos.le_next;
		break;

	default:
		/* return the matching item_id info record */
		for (info = subj->aud_info_head.lh_first;
		    info != NULL; info = info->aud_infos.le_next) {
			if (info->item_id == item_id)
				break;
		}
	}

	if (!info) {
		errno = EINVAL;
		return(-1);
	}

	subj->aud_info_cur = info;
	aud_subj_info_p->aud_info_type = info->aud_info_type;
	aud_subj_info_p->aud_info_length = info->aud_info_length;
	aud_subj_info_p->aud_info_p = info->aud_info_p;
        /* XXXX non-standard, but useful for aud_dup_record */
        aud_subj_info_p->item_id = info->item_id;    
	return(subj->aud_info_num);
}


/*
 * XXXXXX Should probably use getpwnam -- but that is static?
 */
aud_id_t
aud_id_from_text(const char *text_p)
{

	errno = EINVAL;
	return((aud_id_t)-1);
}


/*
 * XXXXXX Should probably use getpwuid -- but that is static?
 */
char
*aud_id_to_text(aud_id_t audit_ID, ssize_t *len_p)
{

	errno = EINVAL;
	return((char *)NULL);
}


/*
 * aud_init_record: allocate a new struct aud_rec_t, initialize appropriately,
 * register with object manager, and return a pointer to it.
 */
aud_rec_t
aud_init_record(void)
{
	struct aud_rec_str	*myrec;
	int	i;

	/* Allocate and register type */

	myrec = (struct aud_rec_str *) malloc(sizeof(struct aud_rec_str));
	if (!myrec) {
		errno = ENOMEM;
		return((aud_rec_t)NULL);
	}
	myrec->magic = AUD_MAGIC_AUDREC;

	i = audit_private_newobject(AUD_TYPE_IMP_AUD_REC, myrec);
	if (i) {
		free(myrec);
		return((aud_rec_t)NULL);
	}

	/* Initialize structure */

	myrec->aud_num_hdr = 0;
	LIST_INIT(&myrec->aud_hdr_head);

	myrec->aud_num_subj = 0;
	LIST_INIT(&myrec->aud_subj_head);

	myrec->aud_num_evinfo = 0;
	LIST_INIT(&myrec->aud_evinfo_head);

	myrec->aud_num_obj = 0;
	LIST_INIT(&myrec->aud_obj_head);

	return((aud_rec_t)myrec);
}


/*
 * Create a new aud_evinfo_str in the audit record, optionally before an
 * existing record specified via next_p, and return a descriptor for the
 * aud_evinfo_str.
 */
int
aud_put_event(aud_rec_t ar, const aud_evinfo_t *next_p,
	      aud_evinfo_t *new_p)
{
	struct aud_rec_str	*rec;
	struct aud_evinfo_str	*old, *pred, *new;

	rec = ar;
	if (rec->magic != AUD_MAGIC_AUDREC) {
		errno = EINVAL;
		return(-1);
	}

	if (next_p)
		old = *next_p;
	else
		old = NULL;

	if (old && old->magic != AUD_MAGIC_EVINFO) {
		errno = EINVAL;
		return(-1);
	}

	/*
	 * find the predecessor in the list for our new record before 
	 * malloc'ing to save rolling back
	 */
	if (old) {
		for (pred = rec->aud_evinfo_head.lh_first; pred != NULL;
                    pred = pred->aud_evinfos.le_next) {
                        if (pred == old)
				break;
		}
		if (pred == NULL) {
			errno = EINVAL;
			return(-1);
		}
	} else {
		pred = NULL;
	}

	new = (struct aud_evinfo_str *) malloc(sizeof(struct aud_evinfo_str));
	if (new == NULL) {
		errno = ENOMEM;
		return(-1);
	}
	new->magic = AUD_MAGIC_EVINFO;

	new->myrec = rec;
	rec->aud_num_evinfo++;

	new->aud_info_num = 0;
	new->aud_info_cur = NULL;
	LIST_INIT(&new->aud_info_head);

	/*
	 * XXXX for some reason, gdb gets *really* confused by this
	 */
	if (pred == NULL)
		LIST_INSERT_HEAD(&rec->aud_evinfo_head, new, aud_evinfos);
	else
		LIST_INSERT_BEFORE(pred, new, aud_evinfos);

	*new_p = new;

	return(0);
}


/*
 * Add an info to an evinfo; item_id should/must be unique, position
 * indicates either a) the item to put it before, or b) AUD_LAST_ITEM to
 * indicate appending.
 */
int
aud_put_event_info(aud_evinfo_t aud_event_d, int position, int item_id,
		   const aud_info_t *aud_event_info_p)
{
	struct aud_evinfo_str	*evinfo;
	aud_info_t	*insertbefore, *info, *new, *tail;

	evinfo = (struct aud_evinfo_str *) aud_event_d;
	if (evinfo->magic != AUD_MAGIC_EVINFO) {
		errno = EINVAL;
		return(-1);
	}

	/* 
	 * scan through list, looking for item_id, and also position if
	 * position != AUD_LAST_ITEM.  If item_id is found, return EINVAL;
	 * if position is found, keep a pointer to it and keep traversing
	 * to assure that we do not run into a matching item_id.  If position
	 * is not found, return EINVAL.
	 */
	insertbefore = NULL;
	tail = NULL;
	for (info = evinfo->aud_info_head.lh_first; info != NULL;
	    info = info->aud_infos.le_next) {
		if (info->item_id == item_id) {
			errno = EINVAL;
			return(-1);
		}
		if (position != AUD_LAST_ITEM)
			if (position == info->item_id)
				insertbefore = info;
		tail = info;
	}

	if ((position != AUD_LAST_ITEM) && (insertbefore == NULL)) {
		errno = EINVAL;
		return(-1);
	}

	/*
	 * make a copy of the info
	 */
	if (!aud_event_info_p->aud_info_length) {
		errno = EINVAL;
		return(-1);
	}

	new = (aud_info_t *) malloc(sizeof(aud_info_t));
	if (new == NULL) {
		errno = ENOMEM;
		return(-1);
	}
	new->aud_info_type = aud_event_info_p->aud_info_type;
	new->aud_info_length = aud_event_info_p->aud_info_length;
	new->item_id = item_id;
	new->aud_info_p = malloc(new->aud_info_length);

	if (new->aud_info_p == NULL) {
		free(new);
		errno = ENOMEM;
		return(-1);
	}

	memcpy(new->aud_info_p, aud_event_info_p->aud_info_p,
	    new->aud_info_length);

	if (insertbefore)
		LIST_INSERT_BEFORE(insertbefore, new, aud_infos);
	else {
		if (tail == NULL)
			LIST_INSERT_HEAD(&evinfo->aud_info_head,
			    new, aud_infos);
		else
			LIST_INSERT_AFTER(tail, new, aud_infos);
	}
	evinfo->aud_info_num++;

	return(0);
}


int
aud_put_hdr(aud_rec_t ar, const aud_hdr_t *next_p,
	    aud_hdr_t *new_p)
{
	struct aud_rec_str	*rec;
	struct aud_hdr_str	*old, *pred, *new;

	rec = ar;
	if (rec->magic != AUD_MAGIC_AUDREC) {
		errno = EINVAL;
		return(-1);
	}

	if (next_p)
		old = *next_p;
	else
		old = NULL;

	if ((old != NULL) && (old->magic != AUD_MAGIC_HDR)) {
		errno = EINVAL;
		return(-1);
	}

	/*
	 * find the predecessor in the list for our new record before
	 * malloc'ing to save rolling back
	 */
	if (old) {
		for (pred = rec->aud_hdr_head.lh_first; pred != NULL;
		    pred = pred->aud_hdrs.le_next) {
			if (pred == old)
				break;
		}
		if (pred == NULL) {
			errno = EINVAL;
			return(-1);
		}
	} else {
		pred = NULL;
	}

	new = (struct aud_hdr_str *) malloc(sizeof(struct aud_hdr_str));
	if (new == NULL) {
		errno = ENOMEM;
		return(-1);
	}

	new->magic = AUD_MAGIC_HDR;

	new->myrec = rec;
	rec->aud_num_hdr++;

	new->aud_info_num = 0;
	new->aud_info_cur = NULL;
	LIST_INIT(&new->aud_info_head);

	/*
	 * XXXX for some reason, gdb gets *really* confused by this
	 */
	if (pred == NULL)
		LIST_INSERT_HEAD(&rec->aud_hdr_head, new, aud_hdrs);
	else
		LIST_INSERT_BEFORE(pred, new, aud_hdrs);

	*new_p = new;

	return(0);
}


/*
 * Add an info to a hdr; item_id should/must be unique, position
 * indicates either a) the item to put it before, or b) AUD_LAST_ITEM to
 * indicate appending.
 */
int
aud_put_hdr_info(aud_hdr_t aud_hdr_d, int position, int item_id,
		 const aud_info_t *aud_hdr_info_p)
{
	struct aud_hdr_str	*hdr;
	aud_info_t	*insertbefore, *info, *new, *tail;

	hdr = (struct aud_hdr_str *) aud_hdr_d;
	if (hdr->magic != AUD_MAGIC_HDR) {
		errno = EINVAL;
		return(-1);
	}

	/*
	 * scan through list, looking for item_id, and also position if
	 * position != AUD_LAST_ITEM.  If item_id is found, return EINVAL;
	 * if position is found, keep a pointer to it and keep traversing
	 * to assure that we do not run into a matching item_id.  If position
	 * is not found, return EINVAL.
	 */
	insertbefore = NULL;
	tail = NULL;
	for (info = hdr->aud_info_head.lh_first; info != NULL;
	    info = info->aud_infos.le_next) {
		if (info->item_id == item_id) {
			errno = EINVAL;
			return(-1);
		}
		if (position != AUD_LAST_ITEM)
			if (position == info->item_id)
				insertbefore = info;
		tail = info;
	}

	if ((position != AUD_LAST_ITEM) && (insertbefore == NULL)) {
		errno = EINVAL;
		return(-1);
	}

	/*
	 * make a copy of the info
	 */
	if (!aud_hdr_info_p->aud_info_length) {
		errno = EINVAL;
		return(-1);
	}

	new = (aud_info_t *) malloc(sizeof(aud_info_t));
	if (new == NULL) {
		errno = ENOMEM;
		return(-1);
	}
	new->aud_info_type = aud_hdr_info_p->aud_info_type;
	new->aud_info_length = aud_hdr_info_p->aud_info_length;
	new->item_id = item_id;

	new->aud_info_p = malloc(new->aud_info_length);
	if (new->aud_info_p == NULL) {
		free(new);
		errno = ENOMEM;
		return(-1);
	}

	memcpy(new->aud_info_p, aud_hdr_info_p->aud_info_p,
	    new->aud_info_length);

	if (insertbefore)
		LIST_INSERT_BEFORE(insertbefore, new, aud_infos);
	else {
		if (tail == NULL)
			LIST_INSERT_HEAD(&hdr->aud_info_head, new, aud_infos);
		else
			LIST_INSERT_AFTER(tail, new, aud_infos);
	}
	hdr->aud_info_num++;

	return(0);
}


int
aud_put_obj(aud_rec_t ar, const aud_obj_t *next_p, aud_obj_t *new_p)
{
	struct aud_rec_str	*rec;
	struct aud_obj_str	*old, *pred, *new;

	rec = ar;
	if (rec->magic != AUD_MAGIC_AUDREC) {
		errno = EINVAL;
		return(-1);
	}

	if (next_p)
		old = *next_p;
	else
		old = NULL;

	if (old && old->magic != AUD_MAGIC_OBJ) {
		errno = EINVAL;
		return(-1);
	}

	/*
	 * find the predecessor in the list for our new record before 
	 * malloc'ing to save rolling back
	 */
	if (old) {
		for (pred = rec->aud_obj_head.lh_first; pred != NULL;
		    pred = pred->aud_objs.le_next) {
			if (pred == old)
				break;
		}
		if (pred == NULL) {
			errno = EINVAL;
			return(-1);
		}
	} else {
		pred = NULL;
	}

	new = (struct aud_obj_str *) malloc(sizeof(struct aud_obj_str));
	if (new == NULL) {
		errno = ENOMEM;
		return(-1);
	}
	new->magic = AUD_MAGIC_OBJ;

	new->myrec = rec;
	rec->aud_num_obj++;

	new->aud_info_num = 0;
	new->aud_info_cur = NULL;
	LIST_INIT(&new->aud_info_head);

	/*
	 * XXXX for some reason, gdb gets *really* confused by this
	 */

	if (pred == NULL)
		LIST_INSERT_HEAD(&rec->aud_obj_head, new, aud_objs);
        else
		LIST_INSERT_BEFORE(pred, new, aud_objs);

	*new_p = new;

	return(0);
}


/*
 * Add an info to an obj; item_id should/must be unique, position
 * indicates either a) the item to put it before, or b) AUD_LAST_ITEM to
 * indicate appending.
 */
int
aud_put_obj_info(aud_obj_t aud_obj_d, int position, int item_id,
		 const aud_info_t *aud_obj_info_p)
{
	struct aud_obj_str	*obj;
	aud_info_t	*insertbefore, *info, *new, *tail;

	obj = (struct aud_obj_str *) aud_obj_d;
	if (obj->magic != AUD_MAGIC_OBJ) {
		errno = EINVAL;
		return(-1);
	}

	/*
	 * scan through list, looking for item_id, and also position if
	 * position != AUD_LAST_ITEM.  If item_id is found, return EINVAL;
	 * if position is found, keep a pointer to it and keep traversing
	 * to assure that we do not run into a matching item_id.  If position
	 * is not found, return EINVAL.
	 */
	insertbefore = NULL;
	tail = NULL;
	for (info = obj->aud_info_head.lh_first; info != NULL;
	    info = info->aud_infos.le_next) {
		if (info->item_id == item_id) {
			errno = EINVAL;
			return(-1);
		}
		if (position != AUD_LAST_ITEM)
			if (position == info->item_id)
				insertbefore = info;
		tail = info;
	}

	if ((position != AUD_LAST_ITEM) && (insertbefore == NULL)) {
		errno = EINVAL;
		return(-1);
	}

	/*
	 * make a copy of the info
	 */
	if (!aud_obj_info_p->aud_info_length) {
		errno = EINVAL;
		return(-1);
	}

	new = (aud_info_t *) malloc(sizeof(aud_info_t));
	if (new == NULL) {
		errno = ENOMEM;
		return(-1);
	}
	new->aud_info_type = aud_obj_info_p->aud_info_type;
	new->aud_info_length = aud_obj_info_p->aud_info_length;
	new->item_id = item_id;
	new->aud_info_p = malloc(new->aud_info_length);

	if (new->aud_info_p == NULL) {
		free(new);
		errno = ENOMEM;
		return(-1);
	}

	memcpy(new->aud_info_p, aud_obj_info_p->aud_info_p,
	    new->aud_info_length);

	if (insertbefore)
		LIST_INSERT_BEFORE(insertbefore, new, aud_infos);
	else {
		if (tail == NULL)
			LIST_INSERT_HEAD(&obj->aud_info_head,
			    new, aud_infos);
		else
			LIST_INSERT_AFTER(tail, new, aud_infos);
	}
	obj->aud_info_num++;

	return(0);
}


/*
 * Create a new aud_subj_str in the audit record, optionally before an
 * existing record specified via next_p, and return a descriptor for the
 * aud_subj_str.
 */
int
aud_put_subj(aud_rec_t ar, const aud_subj_t *next_p,
	     aud_subj_t *new_p)
{
	struct aud_rec_str	*rec;
	struct aud_subj_str	*old, *pred, *new;

	rec = ar;
	if (rec->magic != AUD_MAGIC_AUDREC) {
		errno = EINVAL;
		return(-1);
	}

	if (next_p)
		old = *next_p;
	else
		old = NULL;

	if (old && old->magic != AUD_MAGIC_SUBJ) {
		errno = EINVAL;
		return(-1);
	}

	/*
	 * find the predecessor in the list for our new record before 
	 * malloc'ing to save rolling back
	 */
	if (old) {
		for (pred = rec->aud_subj_head.lh_first; pred != NULL;
                    pred = pred->aud_subjs.le_next) {
                        if (pred == old)
				break;
		}
		if (pred == NULL) {
			errno = EINVAL;
			return(-1);
		}
	} else {
		pred = NULL;
	}

	new = (struct aud_subj_str *) malloc(sizeof(struct aud_subj_str));
	if (new == NULL) {
		errno = ENOMEM;
		return(-1);
	}
	new->magic = AUD_MAGIC_EVINFO;

	new->myrec = rec;
	rec->aud_num_subj++;

	new->aud_info_num = 0;
	new->aud_info_cur = NULL;
	LIST_INIT(&new->aud_info_head);

	/*
	 * XXXX for some reason, gdb gets *really* confused by this
	 */
	if (pred == NULL)
		LIST_INSERT_HEAD(&rec->aud_subj_head, new, aud_subjs);
	else
		LIST_INSERT_BEFORE(pred, new, aud_subjs);

	*new_p = new;

	return(0);
}

/*
 * Add an info to an subj; item_id should/must be unique, position
 * indicates either a) the item to put it before, or b) AUD_LAST_ITEM to
 * indicate appending.
 */
int
aud_put_subj_info(aud_subj_t aud_subj_d, int position, int item_id,
		  const aud_info_t *aud_subj_info_p)
{
	struct aud_subj_str	*subj;
	aud_info_t	*insertbefore, *info, *new, *tail;

	subj = (struct aud_subj_str *) aud_subj_d;
	if (subj->magic != AUD_MAGIC_SUBJ) {
		errno = EINVAL;
		return(-1);
	}

	/* 
	 * scan through list, looking for item_id, and also position if
	 * position != AUD_LAST_ITEM.  If item_id is found, return EINVAL;
	 * if position is found, keep a pointer to it and keep traversing
	 * to assure that we do not run into a matching item_id.  If position
	 * is not found, return EINVAL.
	 */
	insertbefore = NULL;
	tail = NULL;
	for (info = subj->aud_info_head.lh_first; info != NULL;
	    info = info->aud_infos.le_next) {
		if (info->item_id == item_id) {
			errno = EINVAL;
			return(-1);
		}
		if (position != AUD_LAST_ITEM)
			if (position == info->item_id)
				insertbefore = info;
		tail = info;
	}

	if ((position != AUD_LAST_ITEM) && (insertbefore == NULL)) {
		errno = EINVAL;
		return(-1);
	}

	/*
	 * make a copy of the info
	 */
	if (!aud_subj_info_p->aud_info_length) {
		errno = EINVAL;
		return(-1);
	}

	new = (aud_info_t *) malloc(sizeof(aud_info_t));
	if (new == NULL) {
		errno = ENOMEM;
		return(-1);
	}
	new->aud_info_type = aud_subj_info_p->aud_info_type;
	new->aud_info_length = aud_subj_info_p->aud_info_length;
	new->item_id = item_id;
	new->aud_info_p = malloc(new->aud_info_length);

	if (new->aud_info_p == NULL) {
		free(new);
		errno = ENOMEM;
		return(-1);
	}

	memcpy(new->aud_info_p, aud_subj_info_p->aud_info_p,
	    new->aud_info_length);

	if (insertbefore)
		LIST_INSERT_BEFORE(insertbefore, new, aud_infos);
	else {
		if (tail == NULL)
			LIST_INSERT_HEAD(&subj->aud_info_head,
			    new, aud_infos);
		else
			LIST_INSERT_AFTER(tail, new, aud_infos);
	}
	subj->aud_info_num++;

	return(0);
}


/*
 * aud_read_sc: given an open file (filedes), read a new record if available
 * Blocking/non-blocking is hard to deal with; as are records that have
 * not completely arrived yet.  The simplist solution (and is permitted by
 * the spec) is simply not to deal with failure cases.  aud_read_sc returns
 * a uid from the flat_header; this is not needed for most applications, so
 * aud_read is a wrapper that ignores the uid.  uid is probably only useful
 * for auditd
 *
 * Read in flat_header
 * Read in chunk of size indicated by flat_header
 * Convert to an aud_rec_t
 * Return the aud_rec_t
 */
aud_rec_t
aud_read_sc(int filedes, uid_t *uid)
{
	int	i, sofar;
	struct flat_header	fh;
	void	*fh_ptr;
	char	*buf, *cur;
	aud_rec_t	rec;

	/* read in the header */
	sofar = 0;
	fh_ptr = &fh;
	while (sofar < sizeof(struct flat_header)) {
		i = read(filedes, fh_ptr + sofar, sizeof(struct flat_header) -
		    sofar);
		if ((i == -1) && (errno != EINTR))
			return(NULL);
		if (i == -1)
			i = 0;
		sofar += i;
		fh_ptr += i;
	}

	/* have a flat_header */
	if (fh.fh_magic != AUD_MAGIC_FLAT) {
		/* XXXX check this */
		errno = EINVAL;
		return(NULL);
	}

	/* alloc a buffer */
	buf = (char *) malloc(fh.fh_length + sizeof(struct flat_header));
	if (buf == NULL) {
		errno = ENOMEM;
		return(NULL);
	}

	memcpy(buf, &fh, sizeof(struct flat_header));
	cur = buf + sizeof(struct flat_header);

	sofar = 0;
	while (sofar < fh.fh_length) {
		i = read(filedes, cur, fh.fh_length - sofar);
		if (i == -1) 
			if (errno != EINTR) {
				free(buf);
				return(NULL);
			} else
				i = 0;
		sofar += i;
		cur += i;
	}

	rec = audit_copy_int_np(buf, fh.fh_length +
	    sizeof(struct flat_header));
	if (!rec) {
		free(buf);
		return(NULL);
	}
	
	free(buf);
	*uid = fh.fh_uid;
	return((aud_rec_t)rec);
}


/*
 * Wrapper that ignores the uid from the flat_header; this is the exposed API
 * call to be used by most applications.  See description of aud_read_sc above
 * for more detailed notes.
 */
aud_rec_t
aud_read(int filedes)
{
	uid_t uid;

	return(aud_read_sc(filedes, &uid));
}


static void
audit_private_info_to_text(char *buf, int buflen, aud_info_t *info)
{
	char	*temp;
	int	tempint;

	switch(info->aud_info_type) {
	case AUD_TYPE_ACL:
	case AUD_TYPE_ACL_TYPE:
	case AUD_TYPE_CAP:
	case AUD_TYPE_INF:
	case AUD_TYPE_MAC:
		snprintf(buf, buflen, "(unrecognized type)");
		return;

	case AUD_TYPE_AUD_ID:
		/* aud_id_t == uid_t */
		snprintf(buf, buflen, "uid=%d", *(aud_id_t *)info->aud_info_p);
		return;

	case AUD_TYPE_AUD_OBJ_TYPE:
		/* aud_obj_type_t */
		switch(*(aud_obj_type_t *)info->aud_info_p) {
		case AUD_OBJ_BLOCK_DEV:
			temp = "block device";
			break;
		case AUD_OBJ_CHAR_DEV:
			temp = "character device";
			break;
		case AUD_OBJ_DIR:
			temp = "directory";
			break;
		case AUD_OBJ_FIFO:
			temp = "FIFO object";
			break;
		case AUD_OBJ_FILE:
			temp = "regular file";
			break;
		case AUD_OBJ_PROC:
			temp = "process object";
			break;
		default:
			temp = "(unknown)";
		}

		snprintf(buf, buflen, "object=%s", temp);
		return;

	case AUD_TYPE_AUD_STATE:
		/* aud_status_t */
		switch(*(aud_status_t *)info->aud_info_p) {
		case AUD_STATE_OFF:
			temp = "off";
			break;
		case AUD_STATE_ON:
			temp = "on";
			break;
		case AUD_STATE_QUERY:
			temp = "query";
			break;
		default:
			temp = "(unknown)"; 
		}

		snprintf(buf, buflen, "state=%s", temp);
		return;

	case AUD_TYPE_AUD_STATUS:
		/* aud_state_t */
		switch(*(aud_state_t *)info->aud_info_p) {
		case AUD_FAIL_PRIV:
			temp = "failed appropriate privilege check";
			break;
		case AUD_FAIL_DAC:
			temp = "failed DAC access checks";
			break;
		case AUD_FAIL_MAC:
			temp = "failed MAC access checks";
			break;
		case AUD_FAIL_OTHER:
			temp = "failed for other reasons";
			break;
		case AUD_PRIV_USED:
			temp = "successfully completed with appropriate"
			    " privelege used";
			break;
		case AUD_SUCCESS:
			temp = "successfully completed";
			break;
		default:
			temp = "(unknown)";
		}

		snprintf(buf, buflen, "status=%s", temp);
		return;

	case AUD_TYPE_AUD_TIME:
		/* aud_time_t */
		snprintf(buf, buflen, "time=%ld.%ld",
		    (*(aud_time_t *)info->aud_info_p).sec,
		    (*(aud_time_t *)info->aud_info_p).nsec);
		return;

	case AUD_TYPE_CHAR:
		/* char */
		snprintf(buf, buflen, "char=%c", *(char *)info->aud_info_p);
		return;

	case AUD_TYPE_GID:
		/* gid_t */
		snprintf(buf, buflen, "gid=%d", *(gid_t *)info->aud_info_p);
		return;

	case AUD_TYPE_INT:
		/* int */
		snprintf(buf, buflen, "int=%d", *(int *)info->aud_info_p);
		return;

	case AUD_TYPE_LONG:
		snprintf(buf, buflen, "long=%ld", *(long *)info->aud_info_p);
		return;

	case AUD_TYPE_MODE:
		snprintf(buf, buflen, "mode=%o", *(mode_t *)info->aud_info_p);
		return;

	case AUD_TYPE_OPAQUE:
		snprintf(buf, buflen, "opaque=(opaque)");
		return;

	case AUD_TYPE_PID:
		snprintf(buf, buflen, "pid=%d", *(pid_t *)info->aud_info_p);
		return;

	case AUD_TYPE_SHORT:
		snprintf(buf, buflen, "short=%hd", *(short *)info->aud_info_p);
		return;

	case AUD_TYPE_STRING:
		tempint = MIN(info->aud_info_length + strlen("string="),
		    buflen);
		snprintf(buf, tempint, "string=%s", (char *)info->aud_info_p);
		return;

	case AUD_TYPE_STRING_ARRAY:
		/* XXXXX how to render a string array? */
		snprintf(buf, buflen, "stringarray=(opaque)");
		return;

	case AUD_TYPE_TIME:
		/* time_t */
		snprintf(buf, buflen, "time=%d", *(pid_t *)info->aud_info_p);
		return;

	case AUD_TYPE_UID:
		/* uid_t */
		snprintf(buf, buflen, "uid=%d", *(uid_t *)info->aud_info_p);
		return;

	default:
		/* unknown */
		snprintf(buf, buflen, "(unknown type)");
		return;
	}

}

/*
 * XXXX generate a string (freeable) containing the record
 * return the string len in len_p
 */
char *
aud_rec_to_text(aud_rec_t ar, ssize_t *len_p)
{
	aud_hdr_t	hdr;
	aud_evinfo_t	evinfo;
	aud_obj_t	obj;
	/* aud_subj_t	subj; */
	aud_info_t	info;
#define BUFLEN	120
	char	*string, *oldstring, *tempstring, buf[BUFLEN];
	int	i, aud_num_hdr, aud_num_evinfo, aud_num_obj, aud_num_subj,
		count, stringret;
	int	format_ok=0;
	int	temp;

	aud_num_hdr = aud_get_hdr(ar, 0, NULL);
	if (aud_num_hdr == -1)
		return(NULL);

	aud_num_evinfo = aud_get_event(ar, 0, NULL);
	if (aud_num_evinfo == -1)
		return(NULL);

	aud_num_obj = aud_get_obj(ar, 0, NULL);
	if (aud_num_obj == -1)
		return(NULL);

	aud_num_subj = aud_get_subj(ar, 0, NULL);
	if (aud_num_subj == -1)
		return(NULL);

	stringret = asprintf(&string, "%d header(s), %d evinfos, %d objects, "
	    "%d subjects\n", aud_num_hdr, aud_num_evinfo, aud_num_obj,
	    aud_num_subj);
	if (stringret == -1) {
		errno = ENOMEM;
		return(NULL);
	}

	if (aud_num_hdr > 0) {
		oldstring = string;
		stringret = asprintf(&string, "%sHeader:\n", oldstring);
		free(oldstring);
		if (stringret == -1) {
			errno = ENOMEM;
			return(NULL);
		}

		/*
	 	 * Display the only the first header, and only the expected
		 * items
		 */

		i = aud_get_hdr(ar, 1, &hdr);
		if (i == -1) {
			return(NULL);
		}

		i = aud_get_hdr_info(hdr, AUD_FORMAT, &info);
		if (i != -1) {
			if ((info.aud_info_type == AUD_TYPE_SHORT) &&
			    (info.aud_info_length == sizeof(short)) &&
			    (*(short *)info.aud_info_p == AUD_NATIVE)) {
				format_ok = 1;
			} else {
				oldstring = string;
				stringret = asprintf(&string, "%sUnrecognized "
				    "format\n", oldstring);
				free(oldstring);
				if (stringret == -1) {
					errno = ENOMEM;
					return(NULL);
				}
			}
		} else {
			oldstring = string;
			stringret = asprintf(&string, "%sUnrecognized format\n",
			    oldstring);
			free(oldstring);
			if (stringret == -1) {
				errno = ENOMEM;
				return(NULL);
			}
			return(string);
		}

		if (format_ok) {
			i = aud_get_hdr_info(hdr, AUD_VERSION, &info);
			if ((i != -1) && (info.aud_info_type ==
			    AUD_TYPE_SHORT)) {
				audit_private_info_to_text(buf, BUFLEN, &info);

				oldstring = string;
				stringret = asprintf(&string, "%sVersion: %s\n",
				    oldstring, buf);
				free(oldstring);
				if (stringret == -1) {
					errno = ENOMEM;	
					return(NULL);
				}
			} else {
				oldstring = string;
				stringret = asprintf(&string, "%sVersion: "
				    "Unknown\n", oldstring);
				free(oldstring);
				if (stringret == -1) {
					errno = ENOMEM;
					return(NULL);
				}
			}

			i = aud_get_hdr_info(hdr, AUD_AUD_ID, &info);
			if ((i != -1) && (info.aud_info_type ==
			    AUD_TYPE_AUD_ID)) {
				audit_private_info_to_text(buf, BUFLEN, &info);

				oldstring = string;
				stringret = asprintf(&string, "%sAudit ID: "
				    "%s\n", oldstring, buf);
				free(oldstring);
				if (stringret == -1) {
					errno = ENOMEM;
					return(NULL);
				}
			} else {
				oldstring = string;
				stringret = asprintf(&string, "%sAudit ID: "
				    "None\n", oldstring);
				free(oldstring);
				if (stringret == -1) {
					errno = ENOMEM;
					return(NULL);
				}
			}

			i = aud_get_hdr_info(hdr, AUD_EVENT_TYPE, &info);
			if ((i != -1) && (info.aud_info_type ==
			    AUD_TYPE_INT)) {
				tempstring = aud_evid_to_text(
				    *(int *)info.aud_info_p, &temp);
				
				oldstring = string;
				if (tempstring) 
					stringret = asprintf(&string,
					    "%sEvent type: %s\n", oldstring,
					    tempstring);
				else
					stringret = asprintf(&string,
					    "%sEvent type: None\n", oldstring);
				free(oldstring);
				if (stringret == -1) {
					errno = ENOMEM;
					return(NULL);
				}
			} else if ((i != -1) && (info.aud_info_type ==
			    AUD_TYPE_STRING)) {
				audit_private_info_to_text(buf, BUFLEN, &info);

				oldstring = string;
				stringret = asprintf(&string, "%sEvent type: "
				    "%s\n", oldstring, buf);
				free(oldstring);
				if (stringret == -1) {
					errno = ENOMEM;
					return(NULL);
				}
			} else {
				oldstring = string;
				stringret = asprintf(&string, "%sEvent type: "
				    "None\n", oldstring);
				free(oldstring);
				if (stringret == -1) {
					errno = ENOMEM;
					return(NULL);
				}
			}

			i = aud_get_hdr_info(hdr, AUD_TIME, &info);
			if ((i != -1) && (info.aud_info_type ==
			    AUD_TYPE_AUD_TIME)) {
				audit_private_info_to_text(buf, BUFLEN, &info);

				oldstring = string;
				stringret = asprintf(&string, "%sTime: "
				    "%s\n", oldstring, buf);
				free(oldstring);
				if (stringret == -1) {
					errno = ENOMEM;
					return(NULL);
				}
			} else {
				oldstring = string;
				stringret = asprintf(&string, "%sTime: "
				    "None\n", oldstring);
				free(oldstring);
				if (stringret == -1) {
					errno = ENOMEM;
					return(NULL);
				}
			}

			i = aud_get_hdr_info(hdr, AUD_STATUS, &info);
			if ((i != -1) && (info.aud_info_type ==
			    AUD_TYPE_AUD_STATUS)) {
				audit_private_info_to_text(buf, BUFLEN, &info);

				oldstring = string;
				stringret = asprintf(&string, "%sStatus: "
				    "%s\n", oldstring, buf);
				free(oldstring);
				if (stringret == -1) {
					errno = ENOMEM;
					return(NULL);
				}
			} else {
				oldstring = string;
				stringret = asprintf(&string, "%sStatus: "
				    "None\n", oldstring);
				free(oldstring);
				if (stringret == -1) {
					errno = ENOMEM;
					return(NULL);
				}
			}

			i = aud_get_hdr_info(hdr, AUD_ERRNO, &info);
			if ((i != -1) && (info.aud_info_type ==
			    AUD_TYPE_INT)) {

				oldstring = string;
				stringret = asprintf(&string, "%sErrno: "
				    "%s\n", oldstring, strerror(
				    *(int *)info.aud_info_p));
				free(oldstring);
				if (stringret == -1) {
					errno = ENOMEM;
					return(NULL);
				}
			} else {
				oldstring = string;
				stringret = asprintf(&string, "%sErrno: "
				    "None\n", oldstring);
				free(oldstring);
				if (stringret == -1) {
					errno = ENOMEM;
					return(NULL);
				}
			}
		}
	}

	for (count = 1; count <= aud_num_evinfo; count++) {
		i = aud_get_event(ar, count, &evinfo);
		if (i == -1) {
			free(string);
			return(NULL);
		}

		i = aud_get_event_info(evinfo, AUD_FIRST_ITEM, &info);
		if (i != -1) {
			audit_private_info_to_text(buf, BUFLEN, &info);

			oldstring = string;
			stringret = asprintf(&string, "%sEvent Info: "
			    "%s\n", oldstring, buf);
			free(oldstring);
			if (stringret == -1) {
				errno = ENOMEM;
				return(NULL);
			}
		}
		while (aud_get_event_info(evinfo, AUD_NEXT_ITEM, &info) != -1) {
			audit_private_info_to_text(buf, BUFLEN, &info);

			oldstring = string;
			stringret = asprintf(&string, "%sEvent Info: "
			    "%s\n", oldstring, buf);
			free(oldstring);
			if (stringret == -1) {
				errno = ENOMEM;
				return(NULL);
			}
		}
	}

	for (count = 1; count <= aud_num_obj; count++) {
		i = aud_get_obj(ar, count, &obj);
		if (i == -1) {
			free(string);
			return(NULL);
		}
	}

	for (count = 1; count <= aud_num_subj; count++) {
		i = aud_get_subj(ar, count, &obj);
		if (i == -1) {
			free(string);
			return(NULL);
		}
	}

	return(string);
}


/*
 * Calculate the flattened size of an audit record
 * Do this by recursively walking the audit_record, counting each 
 * sub-record (hdr,subj,evinfo,obj), and then each info inside of them
 */
ssize_t
aud_size(aud_rec_t ar)
{
	struct aud_rec_str	*rec;
	struct aud_evinfo_str	*evinfo;
	struct aud_hdr_str	*hdr;
	struct aud_obj_str	*obj;
	struct aud_subj_str	*subj;
	aud_info_t	*info;
	ssize_t	size;

	rec = ar;
	if (rec->magic != AUD_MAGIC_AUDREC) {
		errno = EINVAL;
		return((ssize_t)-1);
	}

	size = sizeof(struct flat_header);

	/*
	 * Now walk each chain and add up the space
	 */

	for (evinfo = rec->aud_evinfo_head.lh_first; evinfo != NULL;
	    evinfo = evinfo->aud_evinfos.le_next) {
		size += sizeof(struct flat_aud_evinfo);

		for (info = evinfo->aud_info_head.lh_first; info != NULL;
		    info = info->aud_infos.le_next) {
			size += sizeof(struct flat_aud_info);
			size += info->aud_info_length;
		}
	}

	for (hdr = rec->aud_hdr_head.lh_first; hdr != NULL;
	    hdr = hdr->aud_hdrs.le_next) {
		size += sizeof(struct flat_aud_hdr);

		for (info = hdr->aud_info_head.lh_first; info != NULL;
		    info = info->aud_infos.le_next) {
			size += sizeof(struct flat_aud_info);
			size += info->aud_info_length;
		}
	}

	for (obj = rec->aud_obj_head.lh_first; obj != NULL;
	    obj = obj->aud_objs.le_next) {
		size += sizeof(struct flat_aud_obj);

		for (info = obj->aud_info_head.lh_first; info != NULL;
		    info = info->aud_infos.le_next) {
			size += sizeof(struct flat_aud_info);
			size += info->aud_info_length;
		}
	}

	for (subj = rec->aud_subj_head.lh_first; subj != NULL;
	    subj = subj->aud_subjs.le_next) {
		size += sizeof(struct flat_aud_subj);

		for (info = subj->aud_info_head.lh_first; info != NULL;
		    info = info->aud_infos.le_next) {
			size += sizeof(struct flat_aud_info);
			size += info->aud_info_length;
		}
	}

	return(size);
}


aud_state_t
aud_switch(aud_state_t aud_state)
{
	int	i;

	i = aud_switch_sc(&aud_state);
	if (i != 0) {
		return((aud_state_t)-1);
	} else
		return(aud_state);
}


/*
 * validate ar as containing:
 *	- at least one header
 *	  containing an AUD_EVENT_TYPE_ID
 *	  containing an AUD_STATUS_ID
 */
int
aud_valid(aud_rec_t ar)
{
	aud_hdr_t	hdr;
	aud_info_t	info;
	int	i;

	i = aud_get_hdr(ar, 1, &hdr);
	if (i == -1)
		return(-1);

	i = aud_get_hdr_info(hdr, AUD_EVENT_TYPE, &info);
	if (i == -1)
		return(-1);

	if ((info.aud_info_type != AUD_TYPE_STRING) &&
	    (info.aud_info_type != AUD_TYPE_INT)) {
		errno = EINVAL;
		return(-1);
	}

	i = aud_get_hdr_info(hdr, AUD_STATUS, &info);
	if (i == -1)
		return(-1);

	if (info.aud_info_type != AUD_TYPE_AUD_STATUS) {
		errno = EINVAL;
		return(-1);
	}

	return(0);
}


/*
 * aud_write: given an audit record, write it to a log.  
 * If filedes is not AUD_SYSTEM_LOG, call aud_valid() to validate the
 * audit record.  Either way, convert it to raw bytes using aud_copy_ext.
 * Then if not AUD_SYSTEM_LOG, call write to send it to the file descriptor
 * the traditional way; otherwise, send it to the kernel using aud_write_sc.
 */
int
aud_write(int filedes, aud_rec_t ar)
{
	/* struct aud_rec_str	*rec; */
	aud_rec_t	newrec;
	aud_time_t	at;
	aud_hdr_t	hdr;
	aud_info_t	info;
	struct timespec	tp;
	char	*buf;
	int	i, buflen, len, sofar;
	short	ashort;

	newrec = aud_dup_record(ar);
	if (newrec == NULL)
		return(-1);

#define RETURN(x, error) {			\
			aud_free(newrec);	\
			if (error)		\
				errno = error;	\
			return(x);		\
		}

	/* add a time stamp if needed */
	i = aud_get_hdr(newrec, 1, &hdr);
	if (i == -1)
		RETURN(-1, errno);

	i = aud_get_hdr_info(hdr, AUD_FORMAT, &info);
	if (i == -1) {
		/* so add one */
		info.aud_info_type = AUD_TYPE_SHORT;
		info.aud_info_length = sizeof(short);
		ashort = AUD_NATIVE;
		info.aud_info_p = &ashort;
		i = aud_put_hdr_info(hdr, AUD_LAST_ITEM, AUD_FORMAT, &info);
		if (i == -1)
			RETURN(-1, errno);
	}

	i = aud_get_hdr_info(hdr, AUD_VERSION, &info);
	if (i == -1) {
		/* so add one */
		info.aud_info_type = AUD_TYPE_SHORT;
		info.aud_info_length = sizeof(short);
		ashort = AUD_STD_NOT;		/* XXXX Nice if we had a num */
		info.aud_info_p = &ashort;
		i = aud_put_hdr_info(hdr, AUD_LAST_ITEM, AUD_VERSION, &info);
		if (i == -1)
			RETURN(-1, errno);
	}

	i = aud_get_hdr_info(hdr, AUD_TIME, &info);
	if (i == -1) {
		/* so add one */
		info.aud_info_type = AUD_TYPE_AUD_TIME;
		info.aud_info_length = sizeof(aud_time_t);
		i = clock_gettime(CLOCK_REALTIME, &tp);
		if (i == -1)
			RETURN(-1, errno);
		at.sec = tp.tv_sec;
		at.nsec = tp.tv_nsec;
		info.aud_info_p = &at;
		i = aud_put_hdr_info(hdr, AUD_LAST_ITEM, AUD_TIME, &info);
		if (i == -1)
			RETURN(-1, errno);
	}

	/* validate */
	if (aud_valid(newrec) == -1)
		RETURN(-1, errno);

	/* convert to raw bytes */
	buflen = aud_size(newrec);
	if (buflen == -1)
		RETURN(-1, errno);

	buf = (char *) malloc(buflen);
	if (buf == NULL) {
		/* 
		 * This error return is not listed in the spec, but
		 * should be (XXXXX) 
		 */
		RETURN(-1, ENOMEM);
	}

#undef RETURN
#define RETURN(x, error) {			\
			aud_free(newrec);	\
			free(buf);		\
			if (error)		\
				errno = error;	\
			return(x);		\
		}

	len = aud_copy_ext(buf, newrec, buflen);
	if (len == -1) 
		RETURN(-1, errno);

#undef RETURN
#define RETURN(x, error) {			\
			free(buf);		\
			if (error)		\
				errno = error;	\
			return(x);		\
		}

	if (filedes != AUD_SYSTEM_LOG) {
		sofar = 0;
		while (sofar != len) {
			i = write(filedes, buf + sofar, len - sofar);
			if (i == -1) {
				RETURN(-1, errno);
			}
			if (i > 0)
				sofar += i;
		}
	} else {
		/* submit to audit subsystem */
		i = aud_write_sc(buf, len);
		if (i == -1)
			RETURN(-1, errno);
	}

	RETURN(0, 0);
#undef RETURN
}

