/*-
 * Copyright (c) 1999 Robert N M Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$FreeBSD $
 */
/*
 * getfacl -- POSIX.1e utility to extract ACLs from files and directories
 * and send the results to stdout
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/acl.h>
#include <sys/stat.h>
#include <err.h>
#include <stdio.h>
#include <unistd.h>


void
usage(void)
{

	fprintf(stderr, "getfacl [-d] [files ...]\n");
}

int
print_acl(char *path, acl_type_t type)
{
	struct stat	sb;
	acl_t	acl;
	char	*acl_text;
	int	error;

	error = stat(path, &sb);
	if (error == -1) {
		perror(path);
		return(-1);
	}

	printf("#file:%s\n#owner:%d\n#group:%d\n", path, sb.st_uid, sb.st_gid);

	acl = acl_get_file(path, type);
	if (!acl) {
		warn(path);
		return(-1);
	}

	acl_text = acl_to_text(acl, 0);
	if (!acl_text) {
		perror(path);
		return(-1);
	}

	printf(acl_text);

	acl_free(acl_text);

	return(0);
}

int
print_acl_from_stdin(acl_type_t type)
{
	char	pathname[MAXPATHLEN+1];
	int	carried_error = 0;

	pathname[MAXPATHLEN] = '\0';
	while (fgets(pathname, MAXPATHLEN, stdin)) {
		/* remove the \n */
		pathname[strlen(pathname)] = '\0';
		if (print_acl(pathname, type) == -1) {
			carried_error = -1;
		}
	}

	return(carried_error);
}

int
main(int argc, char *argv[])
{
	acl_type_t	type = ACL_TYPE_ACCESS;
	int	carried_error = 0;
	int	ch, error, i;

	while ((ch = getopt(argc, argv, "d")) != -1)
		switch(ch) {
		case 'd':
			type = ACL_TYPE_DEFAULT;
			break;
		default:
			usage();
			return(-1);
		}
	argc -= optind;
	argv += optind;


	if (argc == 0) {
		error = print_acl_from_stdin(type);
		return(error);
	}

	for (i = 0; i < argc; i++) {
		if (!strcmp(argv[i], "-")) {
			error = print_acl_from_stdin(type);
			if (error == -1)
				carried_error = -1;
		} else {
			error = print_acl(argv[i], type);
			if (error == -1)
				carried_error = -1;
		}
	}

	return(carried_error);
}
