/*-
 * Copyright (c) 1999 Robert N. M. Watson
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id $
 */

/*
 * POSIX.1e ACL kernel syscalls and supporting routines
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/sysproto.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/vnode.h>
#include <sys/lock.h>
#include <sys/namei.h>
#include <sys/file.h>
#include <sys/proc.h>
#include <sys/sysent.h>
#include <sys/errno.h>
#include <sys/acl.h>

#include "opt_posix.h"

static MALLOC_DEFINE(M_ACL, "acl", "access control list");

#ifdef POSIX1E_ACL

static int
acl_check(struct acl *acl);

static int
vacl_set_acl(struct proc *p, struct vnode *vp, acl_type_t type,
	     struct acl *aclp);

static int
vacl_get_acl(struct proc *p, struct vnode *vp, acl_type_t type,
	     struct acl *aclp);


/*
 * Routine to check the validity of an ACL structure in O(n), assuming
 * presorted.  We require userland to submit only sorted ACLs to the
 * kernel for its calls.
 *
 * Algorithm: walk down the ACL expecting the following order of items
 *	Exactly one ACL_USER_OBJ
 *	Exactly one ACL_GROUP_OBJ
 *	Exactly one ACL_OTHER
 * and optionally:
 *	Exactly one ACL_MASK
 *      Zero or more ACL_USER, sorted by increasing uid
 *	Zero or more ACL_GROUP, sorted by increasing gid
 *
 * Returns 0 on success, EINVAL on failure
 * If whichp is non-null, the bad acl entry index will be returned via
 * *whichp, or -1 if an item is missing.  *whichp is unmodified on 
 * success.
 */
static int
acl_check(struct acl *acl)
{
	struct acl_entry	*entry; 	/* current entry */
	uid_t	obj_uid=-1, obj_gid=-1, highest_uid=0, highest_gid=0;
	int	stage = ACL_USER_OBJ;
	int	i = 0;
	int	count_user_obj=0, count_user=0, count_group_obj=0,
		count_group=0, count_mask=0, count_other=0;
	void	*dummy;

/*
 * The following forces variables to be assigned stack space and hence
 * be accessible in the debugger
 */
/*
	dummy = &i;
	dummy = &acl;
	dummy = &count_other;
	dummy = &stage;
	dummy = &count_group;
	dummy = &count_user;
	dummy = &dummy;
	dummy = &obj_uid;
	dummy = &obj_gid;
	dummy = &highest_uid;
	dummy = &highest_gid;
	dummy = &entry;
*/

	printf("acl_check: checking acl with %d entries\n", acl->acl_cnt);
	while (i < acl->acl_cnt) {

		entry = &acl->acl_entry[i];

		if ((entry->ae_perm | ACL_PERM_BITS) != ACL_PERM_BITS)
			return(EINVAL);

		switch(entry->ae_tag) {
		case ACL_USER_OBJ:
			printf("acl_check: %d: ACL_USER_OBJ\n", i);
			if (stage > ACL_USER_OBJ)
				return(EINVAL);
			stage = ACL_USER;
			count_user_obj++;
			obj_uid = entry->ae_id;
			break;
	
		case ACL_USER:
			printf("acl_check: %d: ACL_USER\n", i);
			if (stage > ACL_USER)
				return(EINVAL);
			stage = ACL_USER;
			if (entry->ae_id == obj_uid)
				return(EINVAL);
			if (count_user && (entry->ae_id <= highest_uid))
				return(EINVAL);
			highest_uid = entry->ae_id;
			count_user++;
			break;	
	
		case ACL_GROUP_OBJ:
			printf("acl_check: %d: ACL_GROUP_OBJ\n", i);
			if (stage > ACL_GROUP_OBJ)
				return(EINVAL);
			stage = ACL_GROUP;
			count_group_obj++;
			obj_gid = entry->ae_id;
			break;
	
		case ACL_GROUP:
			printf("acl_check: %d: ACL_GROUP\n", i);
			if (stage > ACL_GROUP)
				return(EINVAL);
			stage = ACL_GROUP;
			if (entry->ae_id == obj_gid)
				return(EINVAL);
			if (count_group && (entry->ae_id <= highest_gid))
				return(EINVAL);
			highest_gid = entry->ae_id;
			count_group++;
			break;
			
		case ACL_MASK:
			printf("acl_check: %d: ACL_MASK\n", i);
			if (stage > ACL_MASK)
				return(EINVAL);
			stage = ACL_MASK;
			count_mask++;
			break;
	
		case ACL_OTHER:
			printf("acl_check: %d: ACL_OTHER\n", i);
			if (stage > ACL_OTHER)
				return(EINVAL);
			stage = ACL_OTHER;
			count_other++;
			break;
	
		default:
			printf("acl_check: %d: INVALID\n", i);
			return(EINVAL);
		}
		i++;
	}

	if (count_user_obj != 1)
		return(EINVAL);
	
	if (count_group_obj != 1)
		return(EINVAL);

	if (count_mask != 0 && count_mask != 1)
		return(EINVAL);

	if (count_other != 1)
		return(EINVAL);

	return(0);
}


/*
 * acl_attr_to_acl -- for file systems not supporting ACLs directly, this
 * routine may be used to fill in an ACL record reflecting the permissions
 * returned by getattr.  Below, we use the stat(2) permission values,
 * but really this is probably not correct.  They should be converted using
 * the ACL constants, etc. XXX
 */
void
acl_attr_to_acl(struct acl *a_acl, struct vattr *vattr)
{

	bzero(a_acl, sizeof(struct acl));
	a_acl->acl_cnt = 3;
	/* owner */
	a_acl->acl_entry[0].ae_tag = ACL_USER_OBJ
	a_acl->acl_entry[0].ae_id = vattr->va_uid;
	/*
	 * XXX this assumes continuity between types and permissions,
	 * which should not be assumed.
	 */
	a_acl->acl_entry[0].ae_perm = ((vattr->va_mode & S_RWXU) >> 6);
	/* group */
	a_acl->acl_entry[1].ae_tag = ACL_GROUP_OBJ;
	a_acl->acl_entry[1].ae_id = vattr->va_gid;
	/*
	 * XXX this assumes continuity between types and permissions,
	 * which should not be assumed.
	 */
	a_acl->acl_entry[1].ae_perm = ((vattr->va_mode & S_RWXG) >> 3);
	/* other */
	a_acl->acl_entry[2].ae_tag = ACL_OTHER;
	a_acl->acl_entry[2].ae_id = 0;
	/*
	 * XXX this assumes continuity between types and permissions,
	 * which should not be assumed.
	 */
	a_acl->acl_entry[2].ae_perm = ((vattr->va_mode & S_RWXO));
}


/*
 * generic_vop_getacl -- a generic getacl routine that calls the vp's
 * getattr and fakes up an ACL based on the file system mode/owners/etc.
 * Intended to be called by fs's that don't understand ACLs, such as UFS.
 */
int
generic_vop_getacl(struct vnode *vp, acl_type_t type, struct acl *aclp,
		   struct ucred *cred, struct proc *p)
{
	struct	vattr	*vattr;
	int	error;

	error = VOP_GETATTR(vp, vattr, cred, p);
	if (error)
		return(error);

	acl_attr_to_acl(aclp, vattr);

	return(0);
}


/*
 * acl_access -- similar to vop_access except it takes an ACL not a vnode.
 * this call is intended to be used by file systems that store ACLs in
 * the ACL format and want to use a common ACL evaluation routine.  The
 * ACL is assumed to be in sorted format (i.e., would be accepted by the
 * acl_check routine above) so as to facilitate evaluation.
 *
 * Returns 0 on success, an errno on failure.
 */
int
acl_access(struct acl *a_acl, int a_mode, struct ucred *a_cred,
	   struct proc *a_p)
{
	struct acl_entry	*matched_entry = 0;
	struct acl_entry	*mask = 0;
	struct acl_entry	*entry = 0;
	gid_t	*gp;
	int	ei, gi;

	/*
	 * If effective uid is root, in generall the access request should
	 * be granted.  However, if the request includes execute, then
	 * we should check to make sure at least one entry in the ACL is
	 * granted execute request.
	 *
	 * XXX -- should this apply ACL_MASK to the entries it finds to
	 * check to see if the execute bit is effective?
	 * XXX -- this is wrong -- doesn't allow for an ACL_MASK entry
	 * or order evaluation.
	 */
	if (a_cred->cr_uid == 0) {
		if (a_mode & 00100) {
			for (ei = 0; ei < a_acl->acl_cnt; ei++)
				if (a_acl->acl_entry[ei].ae_perm &
				    ACL_PERM_EXEC)
					return(0);
			return(EACCES);
		}
		return(0);
	}

	/*
	 * For all other uid's, walk the entire ACL, storing the first
	 * matched ACL entry, as well as the ACL_MASK entry if one is found.
	 * Because we require sorted order for the ACL, we can assume that
	 * the entries are in the correct order of evaluation as specified
	 * in POSIX.1e, i.e.:
	 * ACL_USER_OBJ
	 */

	ei = 0;
	while (ei < a_acl->acl_cnt) {
		entry = &a_acl->acl_entry[ei];

		switch(entry->ae_tag) {
		case ACL_USER_OBJ:
		case ACL_USER:
			if (!matched_entry)
				if (entry->ae_id == a_cred->cr_uid)
					matched_entry = entry;
			break;

		case ACL_GROUP_OBJ:
		case ACL_GROUP:
			if (!matched_entry)
				for (gi = 0, gp = a_cred->cr_groups;
				    gi < a_cred->cr_ngroups; gi++, gp++)
					if (entry->ae_id == *gp)
						matched_entry = entry;
			break;
		case ACL_MASK:
			mask = entry;

		case ACL_OTHER:
			if (!matched_entry)
				matched_entry = entry;

		default:
			panic("acl_access: invalid ae_tag");
		}
		ei++;
	}

	if (!matched_entry) {
		printf("WARNING: acl_access reported no matched entry\n");
		return(-1);
	}

	if (matched_entry->ae_perm & a_mode != a_mode)
		return(EACCES);

	if ((matched_entry->ae_tag == ACL_USER_OBJ) || 
	    (matched_entry->ae_tag == ACL_OTHER))
		return(0);

	if (!mask)
		return(0);

	if (mask->ae_perm & a_mode == a_mode)
		return(0);

	return(EACCES);
}


/*
 * Real acl set/get from userland, to be wrapped by versions that convert
 * fd's and path's to vnodes.  These calls assume the acl argument is
 * coming from userland, so should not be made from within the kernel.
 * instead, the VOP_{GET,SET}ACL should be invoked by kernel code after
 * acquiring a vnode reference + appropriate locks (see below).
 */
static int
vacl_set_acl(struct proc *p, struct vnode *vp, acl_type_t type,
             struct acl *aclp)
{
	struct acl	inkernacl;
	int	error;

	error = copyin(aclp, &inkernacl, sizeof(struct acl));
	if (error)
		return(error);

	error = acl_check(&inkernacl);
	if (error)
		return(error);

	VOP_LEASE(vp, p, p->p_ucred, LEASE_WRITE);
	vn_lock(vp, LK_EXCLUSIVE | LK_RETRY, p);
	error = VOP_SETACL(vp, type, &inkernacl, p->p_ucred, p);
	VOP_UNLOCK(vp, 0, p);

	return(error);
}


static int
vacl_get_acl(struct proc *p, struct vnode *vp, acl_type_t type,
             struct acl *aclp)
{
	struct acl	inkernelacl;
	int	error;

	error = VOP_GETACL(vp, type, &inkernelacl, p->p_ucred, p);

	if (error == 0) {
		error = copyout(&inkernelacl, aclp, sizeof(struct acl));
		if (error)
			return(error);
	}

	return(error);
}


static int
vacl_delete_def(struct proc *p, struct vnode *vp)
{
	int	error;

	VOP_LEASE(vp, p, p->p_ucred, LEASE_WRITE);
	vn_lock(vp, LK_EXCLUSIVE | LK_RETRY, p);
	error = VOP_SETACL(vp, ACL_TYPE_DEFAULT, 0, p->p_ucred, p);
	VOP_UNLOCK(vp, 0, p);

	return(error);
}


/*
 * Syscalls -- the _set_ calls require that acls be submitted in a sorted
 * order by userland, hence having a wrapper library call instead of using
 * the syscalls directly.
 */
int
acl_syscall_get_file(struct proc *p, struct acl_syscall_get_file_args *uap)
{
	struct nameidata	nd;
	int	error;

	NDINIT(&nd, LOOKUP, FOLLOW, UIO_USERSPACE, SCARG(uap, path), p);
	if (error = namei(&nd))
		return(error);

	error = vacl_get_acl(p, nd.ni_vp, SCARG(uap, type), SCARG(uap, aclp));

	vrele(nd.ni_vp);

	return(error);
}


int
acl_syscall_set_file(struct proc *p, struct acl_syscall_set_file_args *uap)
{
	struct nameidata	nd;
	int	error;

	NDINIT(&nd, LOOKUP, FOLLOW, UIO_USERSPACE, SCARG(uap, path), p);
	if (error = namei(&nd))
		return(error);

	error = vacl_set_acl(p, nd.ni_vp, SCARG(uap, type), SCARG(uap, aclp));

	vrele(nd.ni_vp);

	return(error);
}


int
acl_syscall_get_fd(struct proc *p, struct acl_syscall_get_fd_args *uap)
{
	struct file	*fp;
	int	error;

	if (error = getvnode(p->p_fd, SCARG(uap, filedes), &fp))
		return(error);

	return vacl_get_acl(p, (struct vnode *)fp->f_data, SCARG(uap, type),
	    SCARG(uap, aclp));
}


int
acl_syscall_set_fd(struct proc *p, struct acl_syscall_set_fd_args *uap)
{
	struct file	*fp;
	int	error;

	if (error = getvnode(p->p_fd, SCARG(uap, filedes), &fp))
		return(error);

	return vacl_set_acl(p, (struct vnode *)fp->f_data, SCARG(uap, type),
	    SCARG(uap, aclp));
}


int
acl_syscall_delete_def_file(const char *path_p)
{
	struct nameidata	nd;
	int	error;

	NDINIT(&nd, LOOKUP, FOLLOW, UIO_USERSPACE, SCARG(uap, path), p);
	if (error = namei(&nd))
		return(error);

	error = vacl_delete_def(p, nd.ni_vp);

	vrele(nd.ni_vp);

	return(error);
}


int
acl_syscall_delete_def_fd(int filedes)
{
	struct file	*fp;
	struct vnode	*vp;
	int	error;

	if (error = getvnode(p->p_fd, SCARG(uap, filedes), &fp))
		return(error);

	error = vacl_delete_def(p, (struct vnode *)fp->f_data);

	return(error);
}


#else /* POSIX1E_ACL */

/*
 * Dummy syscalls that only return EOPNOTSUPP
 */

int
acl_syscall_get_file(struct proc *p, struct acl_syscall_get_file_args *uap)
{

	return(EOPNOTSUPP);
}

int
acl_syscall_set_file(struct proc *p, struct acl_syscall_set_file_args *uap)
{

	return(EOPNOTSUPP);
}

int
acl_syscall_get_fd(struct proc *p, struct acl_syscall_get_fd_args *uap) 
{

	return(EOPNOTSUPP);
} 

int
acl_syscall_set_fd(struct proc *p, struct acl_syscall_set_fd_args *uap)
{

	return(EOPNOTSUPP);
}

int
acl_syscall_delete_def_file(const char *path_p)
{

	return(EOPNOTSUPP);
}


int
acl_syscall_delete_def_fd(int filedes)
{

	return(EOPNOTSUPP);
}


#endif
